//===-- matcher/Context.h - Maintain shadow context ------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file was developed by Reid Spencer based on the original SymbolTable.h
// written by the LLVM research group and re-written by Reid Spencer.
// It is distributed under the University of Illinois Open Source License. 
// See autoconf/LICENSE.TXT for details.
//
// [02/10/2006] Adapted for Aspicere2 by Bram Adams.
//
//===----------------------------------------------------------------------===//
//
// This file implements the name/Value symbol table for Aspicere2.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_VALUE_SYMBOL_TABLE_H
#define LLVM_VALUE_SYMBOL_TABLE_H

#include "llvm/Type.h"
#include "llvm/Value.h"
#include <map>

//namespace llvm {

/// This class provides a symbol table of name/value pairs. It is essentially
/// a std::map<std::string,Value*> but has a controlled interface provided by
/// LLVM as well as ensuring uniqueness of names.
///
class Context {

/// @name Types
/// @{
public:

  typedef enum{ACTUAL=0,ARGUMENT,CONSTANT,GLOBAL,CROSSADVICE,TYPE,CROSSFUNCTION} KindOfContext;

  /// @brief A mapping of names to pairs of values and their semantics (actual, argument, constant, ...).
  typedef std::map<const std::string, std::pair<llvm::Value *,KindOfContext> > ValueMap;

  /// @brief An iterator over a ValueMap.
  typedef ValueMap::iterator v_iterator;

  /// @brief A const_iterator over a ValueMap.
  typedef ValueMap::const_iterator const_v_iterator;

  /// @brief A mapping of names to types.
  typedef std::map<const std::string, std::pair<llvm::Type *,KindOfContext> > TypeMap;

  /// @brief An iterator over a TypeMap.
  typedef TypeMap::iterator t_iterator;

  /// @brief A const_iterator over a TypeMap.
  typedef TypeMap::const_iterator const_t_iterator;

/// @}
/// @name Constructors
/// @{
public:

  Context() : LastUnique(0) {}
  ~Context();

/// @}
/// @name Accessors
/// @{
public:

  /// This method finds the value with the given \p name in the
  /// the symbol table. 
  /// @returns the value associated with the \p name
  /// @brief Lookup a named Value.
  llvm::Value *lookupValue(const std::string &name) const;

  /// This method gives the kind of context the value/type with the given \p name 
  ///in the symbol table represents. 
  /// @returns the kind of context of a value/type associated with the \p name; garbage return value if name not registered
  /// @brief Lookup kind of a named Value/Type.
  KindOfContext lookupKind(const std::string &name) const;

  /// This method finds the type with the given \p name in the
  /// the symbol table. 
  /// @returns the type associated with the \p name
  /// @brief Lookup a named Type.
  llvm::Type *lookupType(const std::string &name) const;

  /// Gives index of argument or actual context
  /// @returns index of argument or actual context; bails out and/or returns garbage if relevant variable is no argument/actual
  int index(const std::string &name);

  /// @returns true iff the symbol table is empty
  /// @brief Determine if the symbol table is empty
  inline bool empty() const { return vmap.empty()&&tmap.empty(); }

  /// @brief The number of name/type pairs is returned.
  inline unsigned size() const { return unsigned(vmap.size())+unsigned(tmap.size()); }

  /// Given a base name, return a string that is either equal to it or
  /// derived from it that does not already occur in the symbol table
  /// for the specified type.
  /// @brief Get a name unique to this symbol table
  std::string getUniqueName(const std::string &BaseName) const;

  /// This function can be used from the debugger to display the
  /// content of the symbol table while debugging.
  /// @brief Print out symbol table on stderr
  void dump() const;

/// @}
/// @name Iteration
/// @{
public:

  /// @brief Get an v_iterator that from the beginning of the symbol table.
  inline v_iterator v_begin() { return vmap.begin(); }

  /// @brief Get a const_v_iterator that from the beginning of the symbol table.
  inline const_v_iterator v_begin() const { return vmap.begin(); }

  /// @brief Get an v_iterator to the end of the symbol table.
  inline v_iterator v_end() { return vmap.end(); }

  /// @brief Get a const_v_iterator to the end of the symbol table.
  inline const_v_iterator v_end() const { return vmap.end(); }

  /// @brief Get an t_iterator that from the beginning of the symbol table.
  inline t_iterator t_begin() { return tmap.begin(); }

  /// @brief Get a const_t_iterator that from the beginning of the symbol table.
  inline const_t_iterator t_begin() const { return tmap.begin(); }

  /// @brief Get an t_iterator to the end of the symbol table.
  inline t_iterator t_end() { return tmap.end(); }

  /// @brief Get a const_t_iterator to the end of the symbol table.
  inline const_t_iterator t_end() const { return tmap.end(); }

/// @}
/// @name Mutators
/// @{
public:

  /// This method will strip the symbol table of its names.
  /// @brief Strip the symbol table.
  bool strip();

  /// This method adds the provided value \p N to the symbol table.  The Value
  /// doesn't need to have a name, as the first argument is used to place the value in the symbol table. 
  /// @brief Add a named value to the symbol table
  void insert(const std::string& name,llvm::Value *Val,KindOfContext kind,int index_of_actual_or_argument=0,llvm::Value* stack=0);

  /// This method adds the provided type \p N to the symbol table.  The Type
  /// doesn't need to have a name, as the first argument is used to place the value in the symbol table. 
  /// @brief Add a named value to the symbol table
  void insert(const std::string& name,llvm::Type *Typ,KindOfContext kind=TYPE);

  /// This method removes a value from the symbol table. The first argument
  /// is used to lookup the Value in the
  /// symbol table. If the Value is not in the symbol table, this method
  /// returns false.
  /// @returns true if \p Val was successfully erased, false otherwise
  /// @brief Remove a value from the symbol table.
  bool erase(const std::string& name,llvm::Value* Val);

  /// This method removes a type from the symbol table. The first argument
  /// is used to lookup the Type in the
  /// symbol table. If the Type is not in the symbol table, this method
  /// returns false.
  /// @returns true if \p Typ was successfully erased, false otherwise
  /// @brief Remove a type from the symbol table.
  bool erase(const std::string& name,llvm::Type* Typ);

  /// Given a value and its name vName, remove its existing
  /// entry from the symbol table and insert a new one for Name.
  /// @brief Rename a value in the symbol table
  bool rename(const std::string& vName,llvm::Value *V, const std::string &Name);

  /// Given a type and its name tName, remove its existing
  /// entry from the symbol table and insert a new one for Name.
  /// @brief Rename a type in the symbol table
  bool rename(const std::string& tName,llvm::Type *T, const std::string &Name);

  /**
   * Overwrites stack of variable named vName.
   */
  void addStack(const std::string& vName,llvm::Value* stack);

  llvm::Value* getStack(const std::string& vName) const;

/// @}
/// @name Internal Data
/// @{
private:
  ValueMap vmap;                    ///< The map that holds the symbol table.
  TypeMap tmap;
  mutable int LastUnique; ///< Counter for tracking unique names
  std::map<const std::string,int> name_to_actual_or_argument_index; /**< for ACTUAL, ARGUMENT and CROSSFUNCTION */
  std::map<const std::string,llvm::Value*> name_to_stack; /**< CROSSFUNCTION */

/// @}

};

//} // End llvm namespace

#endif
