//===- weaver/Munger.h - Shadow munger -*- C++ -*-===//
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Aspicere2.
 *
 * The Initial Developer of the Original Code is
 * the Ghislain Hoffman Software Engineering Lab, INTEC, University Ghent.
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Bram Adams <bram_DOT_adams_AT_ugent_DOT_be>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#ifndef MUNGER_H
#define MUNGER_H

#include "matcher/AfterJoinPoint.h"
#include "matcher/AroundJoinPoint.h"
#include "matcher/BeforeJoinPoint.h"
#include "matcher/IntroJoinPoint.h"
#include <set>
//#include "utility/ManagedSet.h"
#include "llvm/Function.h"
#include <iostream>

class Munger{
public:
  virtual void weave(const AfterJoinPoint* jp);
  virtual void weave(const AroundJoinPoint* jp);
  virtual void weave(const BeforeJoinPoint* jp);
  virtual void weave(const IntroJoinPoint* jp);

  virtual const std::string getDescription() const=0;

  virtual ~Munger(){
    //zombie_shadows contains de-parent-ed Value* which need to be deleted
    //    for(std::set<llvm::Value*>::iterator it=zombie_shadows.begin();it!=zombie_shadows.end();it++){
    //      delete *it;
    //    }
  }

  //protected:
  static int getUniqueInteger(){
    return unique_integer++;
  }

  /**
   * Detect and signal whether some around advice prevents proceeding to execution of and hence advising of join point.
   */
  static bool isZombie(llvm::Value* shadow){
    return (zombie_shadows.find(shadow)!=zombie_shadows.end());
  }

  static void addZombie(llvm::Value* shadow){
    zombie_shadows.insert(shadow);
  }

  /**
   * Remove zombies from module.
   */
  static void flushZombies();

  /**
   * Main method body possibly is extracted, which leads to the original "main" named "ASPICERE2_EXTRACTED_..." and an extracted method named "main", and hence the original main is NOT executed (new main is chosen instead). This should be cleaned up at end of weaving process (before optimizations). Also, flush any zombie shadows out there.
   */
  static void cleanUp();

  static void aliasMainMethod(llvm::Function* realMain,llvm::Function* extractedMain){
    _realMain=realMain;
    _extractedMain=extractedMain;
  }

  /**
   * Initializes Munger-system with all cross-advice context found during matching phase.
   */
  static void setCrossAdviceContext(std::multimap<llvm::Value*,std::pair<llvm::Value*,int> >* cCA){
    crossAdviceContext=cCA;
  }

  /**
   * Replace old shadow by new one.
   */
  static void updateCrossAdviceContext(llvm::Value* oldShadow,llvm::Value* newShadow){
    std::pair<std::multimap<llvm::Value*,std::pair<llvm::Value*,int> >::iterator,std::multimap<llvm::Value*,std::pair<llvm::Value*,int> >::iterator> context=crossAdviceContext->equal_range(oldShadow);
    std::vector<std::pair<llvm::Value*,int> > tmp;
    for(std::multimap<llvm::Value*,std::pair<llvm::Value*,int> >::iterator it=context.first;it!=context.second;it++){
      //      std::cout << "MM" << it->first << " -> " << it->second.first << " (" << it->second.second << ")" << std::endl;
      tmp.push_back(it->second);
      //      crossAdviceContext->insert(std::make_pair(newShadow,it->second));
      //      crossAdviceContext->erase(it);
    }
    crossAdviceContext->erase(oldShadow);

    for(std::vector<std::pair<llvm::Value*,int> >::iterator it=tmp.begin();it!=tmp.end();it++){
      crossAdviceContext->insert(std::make_pair(newShadow,*it));
    }
  }

  /*
   * Add extra cross-advice context (e.g. for extracted method bodies).
   */
  static void addCrossAdviceContext(llvm::Value* shadow,llvm::Value* value,int index){
    crossAdviceContext->insert(std::make_pair(shadow,std::make_pair(value,index)));
  }
  
  /**
   * Find all cross-advice context of a particular shadow.
   */
  static std::pair<std::multimap<llvm::Value*,std::pair<llvm::Value*,int> >::iterator,std::multimap<llvm::Value*,std::pair<llvm::Value*,int> >::iterator> getCrossAdviceContext(llvm::Value* shadow){
    //    std::cout << "COUNT:\t" << crossAdviceContext->count(shadow) << std::endl;
    return crossAdviceContext->equal_range(shadow);
  }

  /*
   * Find out if cross-advice context is already packed into a join point struct, and if not do this now.
   */
  static void processCrossAdviceContext(JoinPoint* jp);

  /*
   * Find out if cross-advice context of call to extracted method is already packed into a join point struct, and if not do this now.
   */
  static void processCrossAdviceContextOfCallToExtractedMethod(llvm::Instruction* jp);

  /**
   * Makes clear that there's a join point struct jpStruct constructed for shadow.
   */
  static void crossAdviceContextProcessed(llvm::Value* shadow,llvm::Value* jpStruct){
    processedCrossAdviceContext.insert(std::make_pair(shadow,jpStruct));
  }

  /**
   * Gives join point struct variable for particular shadow if the former is constructed, 0 otherwise.
   */
  static llvm::Value* isCrossAdviceContextProcessed(llvm::Value* shadow){
    std::map<llvm::Value*,llvm::Value*>::iterator it=processedCrossAdviceContext.find(shadow);
    if(it!=processedCrossAdviceContext.end()){
      return it->second;
    }else{
      return 0;
    }
  }

  static void addCallToExtractedMethod(llvm::Instruction* call){
    callsToExtractedMethods.insert(call);
  }

  static void addCallsToExtractedMethod(std::vector<llvm::Instruction*> calls){
    for(unsigned int i=0;i<calls.size();i++) callsToExtractedMethods.insert(calls[i]);
  }

  static std::set<llvm::Instruction*> getCallsToExtractedMethod(){
    return callsToExtractedMethods;
  }

  static bool isCallToExtractedMethod(llvm::Instruction* call){
    std::set<llvm::Instruction*>::iterator it=callsToExtractedMethods.find(call);
    return (it!=callsToExtractedMethods.end());
  }

  /**
   * For after returning we will pioneer marker system:
   * - shadow has context marker, i.e. an instruction following the shadow before which all context initialisation needed for the call to the advice needs to be put
   * - shadow has advice marker, i.e. an instruction following the context marker before which calls to advice need to be put
   */
  static llvm::Instruction* getContextMarker(llvm::Value* shadow){
    std::map<llvm::Value*,std::pair<llvm::Instruction*,llvm::Instruction*> >::iterator it=shadowToMarkers.find(shadow);
    if(it!=shadowToMarkers.end()) return it->second.first;      
    else return 0;
  }

  static llvm::Instruction* getAdviceMarker(llvm::Value* shadow){
    std::map<llvm::Value*,std::pair<llvm::Instruction*,llvm::Instruction*> >::iterator it=shadowToMarkers.find(shadow);
    if(it!=shadowToMarkers.end()) return it->second.second;      
    else return 0;
  }

  static void addMarkers(llvm::Value* shadow,llvm::Instruction* contextMarker,llvm::Instruction* adviceMarker){
    shadowToMarkers.insert(std::make_pair(shadow,std::make_pair(contextMarker,adviceMarker)));
  }

  static void eraseMarkers(llvm::Value* shadow){
    shadowToMarkers.erase(shadow);
  }

  static int nrOfUnusedAdvice(){
    return unusedAdvice.size();
  }

  static void addUnusedAdvice(llvm::Function* f){
    unusedAdvice.insert(f);
  }

  static void reconsiderUnusedAdvice(llvm::Function* f){
    unusedAdvice.erase(f);
  }

  static void flushUnusedAdvice();

  ///OBSOLETE
  static bool isJoinPointStructConstructedYet(llvm::Value* jp){
    std::map<llvm::Value*,std::pair<bool,llvm::Function*> >::iterator it=jpToExistenceOfStruct.find(jp);
    if(it!=jpToExistenceOfStruct.end()) return it->second.first;      
    else return false;
  }

  ///OBSOLETE
  static llvm::Function* getJoinPointStructConstructingFunction(llvm::Value* jp){
    std::map<llvm::Value*,std::pair<bool,llvm::Function*> >::iterator it=jpToExistenceOfStruct.find(jp);
    if(it!=jpToExistenceOfStruct.end()) return it->second.second;      
    else return 0;
  }

  ///OBSOLETE
  static void updateStatusOfJoinPointStructConstruction(llvm::Value* jp,bool newStatus,llvm::Function* constructingFunction){
    jpToExistenceOfStruct[jp]=std::make_pair(newStatus,constructingFunction);
  }

private:
  static int unique_integer;

  //  static std::set<std::auto_ptr<llvm::Value> > zombie_shadows;//absence of proceed() => lower-priority advices don't need to be woven => detect this via zombie_shadows + cleanup in destructor
  //  static ManagedSet<llvm::Value*> zombie_shadows;//absence of proceed() => lower-priority advices don't need to be woven => detect this via zombie_shadows + cleanup in destructor
  static std::set<llvm::Value*> zombie_shadows;//absence of proceed() => lower-priority advices don't need to be woven => detect this via zombie_shadows + cleanup in destructor
  //  static multimap<Value*,Value*> shadow_aliases;//if more than one proceed() => lower-priority advices need to be woven at more than one shadow <--> proceed-method is cloned from originally called Function
  static llvm::Function* _realMain;
  static llvm::Function* _extractedMain;

  static std::map<llvm::Value*,std::pair<llvm::Instruction*,llvm::Instruction*> > shadowToMarkers;
  static std::set<llvm::Function*> unusedAdvice;

  ///OBSOLETE
  static std::map<llvm::Value*,std::pair<bool,llvm::Function*> > jpToExistenceOfStruct;//jp -> [exists?,function_body_in_which_struct_was_created]
  
  static std::multimap<llvm::Value*,std::pair<llvm::Value*,int> >* crossAdviceContext;//owned by Weaver
  static std::map<llvm::Value*,llvm::Value*> processedCrossAdviceContext;//shadow -> jp struct

  static std::set<llvm::Instruction*> callsToExtractedMethods;//contains calls from original (empty) method to extracted method
protected:
  bool init(JoinPoint* jp);//true if further processing is needed; false if no further weaving required
};

#endif
