/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Aspicere2.
 *
 * The Initial Developer of the Original Code is
 * the Ghislain Hoffman Software Engineering Lab, INTEC, University Ghent.
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Bram Adams <bram_DOT_adams_AT_ugent_DOT_be>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "utility/Cloning.h"
#include "utility/Naming.h"
#include "utility/Util.h"
#include "llvm/BasicBlock.h"
#include "llvm/DerivedTypes.h"
#include "llvm/Instruction.h"
#include "llvm/IntrinsicInst.h"
#include "llvm/Module.h"
#include <iostream>

using namespace std;
using namespace llvm;

/**
 * Clone oldFunction's Function signature.
 *
 * @param oldFunction
 * @param ValueMap mapping of oldFunction's Arguments to clone's Arguments
 * @param clone_var_arg_flag just pass the var_arg_clone if true, otherwise an extra argument will be added to the clone (if original Function is vararg) in order to directly pass the va_list through it
 * @return Cloned signature (empty body).
 */
Function* cloneFunctionSignature(const Function* const oldFunction,std::map<Value*, Value*> &ValueMap,bool clone_var_arg_flag,bool add_join_point_struct_in_front){
  std::vector<const Type*> ArgTypes;
  bool is_var_arg=oldFunction->getFunctionType()->isVarArg();
  Module* module=(Module*)(oldFunction->getParent());

  if(add_join_point_struct_in_front) ArgTypes.push_back(PointerType::get(module->getTypeByName(string("struct.JOINPOINT"))));

  for (Function::const_arg_iterator I = oldFunction->arg_begin(), E = oldFunction->arg_end(); I != E; ++I)
    ArgTypes.push_back(I->getType());

  if(!clone_var_arg_flag && is_var_arg){//add i8* argument for va_list
    std::cout << "[EXTRACTER] Adding extra arg for VARARG list..." << endl;
    //    ArgTypes.push_back(PointerType::get(PointerType::get(Type::Int8Ty)));
    ArgTypes.push_back(PointerType::get(Type::Int8Ty));
  }

  // Create a new function type...
  FunctionType *FTy = FunctionType::get(oldFunction->getFunctionType()->getReturnType(),
					ArgTypes, clone_var_arg_flag && is_var_arg);
    
  // Create the new function...
  Function *NewF = new Function(FTy, oldFunction->getLinkage(), oldFunction->getName(),module);
    
  // Loop over the arguments, copying the names of the mapped arguments over...
  Function::arg_iterator DestI = NewF->arg_begin();

  if(add_join_point_struct_in_front) DestI++;//as this is a new argument, it's not used yet in the body, hence it doesn't need to be in the ValueMap

  for (Function::const_arg_iterator I = oldFunction->arg_begin(), E = oldFunction->arg_end(); I != E; ++I){
    DestI->setName(I->getName()); // Copy the name over... DANGER: appended index screws things up possibly! NO, new advice, so args have no name => literally changed
    Value* II=(Value*)&*I;
    ValueMap[II] = DestI++;        // Add mapping to ValueMap
  }

  if(!clone_var_arg_flag && is_var_arg){//name sbyte** argument for va_list
    DestI->setName("myvalist");
  }

  return NewF;
}

llvm::Function* cloneFunctionSignatureFromCall(llvm::CallInst* call,const std::string& name,llvm::Type* returnType,bool addJoinPointStructArgument){
  Function* calledFunction=call->getCalledFunction();
  std::vector<const Type*> ArgTypes;
  std::vector<string> ArgNames;

  /*  for(Function::arg_iterator it=calledFunction->arg_begin();it!=calledFunction->arg_end();it++){
    Type* type=(Type*)it->getType();
    ArgTypes.push_back(type);
    ArgNames.push_back(it->getName());
    }*/

  if(addJoinPointStructArgument){
    Module* module=calledFunction->getParent();
    Type* jp_type=PointerType::get(module->getTypeByName(string("struct.JOINPOINT")));
    ArgTypes.push_back(jp_type);
    ArgNames.push_back(string("jpstruct"));
  }

  for(unsigned int i=1;i<call->getNumOperands();i++){
    Value* op=call->getOperand(i);
    Type* type=(Type*)op->getType();
    ArgTypes.push_back(type);
    ArgNames.push_back(append_number_to_string(string("arg"),i));
  }

  // Create a new function type...
  //  FunctionType *FTy = FunctionType::get(calledFunction->getFunctionType()->getReturnType(),
  FunctionType *FTy = FunctionType::get(returnType,
					ArgTypes, false);
    
  // Create the new function...
  Module* module=(Module*)(calledFunction->getParent());
  Function *NewF = new Function(FTy, calledFunction->getLinkage(), name, module);
    
  // Loop over the arguments, copying the names of the mapped arguments over...
  Function::arg_iterator DestI = NewF->arg_begin();
  for (std::vector<string>::iterator I = ArgNames.begin(); I != ArgNames.end(); ++I){
    DestI->setName(*I); // Copy the name over...
    DestI++;
  }

  return NewF;
}

/// cloneFunctionWithOtherReturnType - Return a copy of the specified function with another return type.
/// Also, any references specified
/// in the ValueMap are changed to refer to their mapped value instead of the
/// original one.  If any of the arguments to the function are in the ValueMap,
/// the arguments are deleted from the resultant function.  The ValueMap is
/// updated to include mappings from all of the instructions and basicblocks in
/// the function from their old to new values.
///
Function * cloneFunctionWithOtherReturnType(const Function *F,Type* returnType,const string& name,
					    DenseMap<const Value*, Value*> &ValueMap,std::vector<pair<string,Type*> >& extraArgs,
					    ClonedCodeInfo *CodeInfo,std::map<llvm::Type*,llvm::Type*>& dummyTypeToContextType,set<string>& typeArgumentNames) {
  std::vector<const Type*> ArgTypes;

  // The user might be deleting arguments to the function by specifying them in
  // the ValueMap.  If so, we need to not add the arguments to the arg ty vector
  //
  int index=0;
  int len=1;
  for (Function::const_arg_iterator I = F->arg_begin(), E = F->arg_end();
       I != E; ++I){
    if (ValueMap.count(I) == 0){  // Haven't mapped the argument to anything yet?
      Type* t=(Type*)I->getType();
      set<string>::iterator nameIt=typeArgumentNames.find(get_name((Argument*)(&*I),len));
      std::map<llvm::Type*,llvm::Type*>::iterator typeIt=dummyTypeToContextType.find(t);
      if((nameIt==typeArgumentNames.end())&&(typeIt!=dummyTypeToContextType.end())){
	ArgTypes.push_back(typeIt->second);//replace dummy type by actual type
      }else{
	ArgTypes.push_back(t);
      }
    }

    index++;
    if(index==10) len=2;
  }

  for(std::vector<pair<string,Type*> >::iterator it=extraArgs.begin();it!=extraArgs.end();it++){
    ArgTypes.push_back(it->second);
  }

  // Create a new function type...
  FunctionType *FTy = FunctionType::get(returnType, ArgTypes, F->getFunctionType()->isVarArg());

  // Create the new function...
  Function *NewF = new Function(FTy, F->getLinkage(), name,(Module*)F->getParent());

  // Loop over the arguments, copying the names of the mapped arguments over...
  Function::arg_iterator DestI = NewF->arg_begin();
  for (Function::const_arg_iterator I = F->arg_begin(), E = F->arg_end();
       I != E; ++I)
    if (ValueMap.count(I) == 0) {   // Is this argument preserved?
      DestI->setName(I->getName()); // Copy the name over... DANGER: problems with indexing? NO, new advice, so args have no name => new name literally set
      ValueMap[I] = DestI++;        // Add mapping to ValueMap
    }

  for(std::vector<pair<string,Type*> >::iterator it=extraArgs.begin();it!=extraArgs.end();it++){
    DestI->setName(it->first);
    DestI++;
  }

  std::vector<ReturnInst*> Returns;  // Ignore returns cloned...
  CloneFunctionInto(NewF, F, ValueMap, Returns, "", CodeInfo);
  return NewF;
}

/**
 * Move oldFunction's BasicBlocks to newFunction using a mapping of oldFunction's Arguments to newFunction's ones.
 */
void transferBasicBlocks(Function* newFunction,Function* oldFunction,std::map<Value*, Value*> &ValueMap){
  //  Module* module=newFunction->getParent();

  // Loop over all of the basic blocks in the function, transfering them as
  // appropriate.
  Function::BasicBlockListType& newBBList=newFunction->getBasicBlockList();
  Function::BasicBlockListType& oldBBList=oldFunction->getBasicBlockList();

  int size=oldBBList.size();
  for (int i=0;i<size;i++){
    BasicBlock &BB = oldBBList.front();

    //add to newFunction
    BB.removeFromParent();
    newBBList.push_back(&BB);

  }

  for(map<Value*, Value*>::iterator it=ValueMap.begin();it!=ValueMap.end();it++){
    Value* oldArg=it->first;
    Value* newArg=it->second;
    oldArg->replaceAllUsesWith(newArg);
  }

}

/**
 * Move function's BasicBlocks to basic's enclosing Function using a mapping of function's Arguments to actuals.
 */
void transferBasicBlocks(Function* function,BasicBlock* basic,std::map<Value*, Value*> &ValueMap){
  // Loop over all of the basic blocks in the function, transfering them as
  // appropriate.
  Function::BasicBlockListType& BBList=function->getBasicBlockList();
  BasicBlock* current=getNextBasicBlock(basic);//possibly unsafe?

  int size=BBList.size();
  for (int i=0;i<size;i++){
    BasicBlock &BB = BBList.front();

    //add to basic's enclosing Function
    BB.moveBefore(current);
    //    current=&BB;
  }

  for(map<Value*, Value*>::iterator it=ValueMap.begin();it!=ValueMap.end();it++){
    Value* oldArg=it->first;
    Value* newArg=it->second;
    oldArg->replaceAllUsesWith(newArg);
  }

}

CallInst* addCallFromTo(Function* from,Function* to){
  //clone entry of to
  //  const BasicBlock& entry=to->getEntryBlock();
  //  std::map<const Value*, Value*> dummy;
  BasicBlock* newEntry=new BasicBlock("",from);//CloneBasicBlock(&entry, dummy, "", from,0);

  const FunctionType* fromType=from->getFunctionType();
  int nr_of_from_args=fromType->getNumParams();
  bool varArg=from->isVarArg();

  const FunctionType* toType=to->getFunctionType();
  bool is_void=(toType->getReturnType()==Type::VoidTy);
  int nr_of_to_args=toType->getNumParams();

  std::cout << "[EXTRACTER] Extracted function " << to->getName() << " is a " << (is_void?"VOID":"non-VOID") << " and " << (varArg?"VARARG":"non-VARARG") << " with " << nr_of_from_args << "/" << nr_of_to_args <<" arguments (original/clone)." << endl;

  //load first N alloca's (of from-Function)
  vector<Value*> actuals;

  Module* module=(Module*)(from->getParent());
  actuals.push_back(Constant::getNullValue(PointerType::get(module->getTypeByName(string("struct.JOINPOINT")))));//for now, just add null (wait until shadow (type)-switch)

  //add from's arguments to actuals of to
  int i=0;
  for(Function::arg_iterator itt=from->arg_begin();itt!=from->arg_end();itt++){
    Instruction* ins=new AllocaInst(itt->getType(),append_number_to_string(string("arg"),i),newEntry);  
    new StoreInst(itt,ins,false,newEntry);
    actuals.push_back(new LoadInst(ins,create_unique_name("tmp"),newEntry));
    i++;
  }

  Instruction* fakeTerminator=new UnreachableInst(newEntry);//dummy terminator for easy manipulation of instructions

  //handle varargs: extract all va_start's from to to from, find the i8* they're using, remove that one's alloca (tied to underlying vararg-construct) and replace its uses by the last argument of the clone; put one va_start here in the delegating from-function (after an alloca for an i8*)
  //    Value* vararg_list=0;
  IntrinsicInst* va_end=0;
  if(varArg){
    //extract all va_start's
    //find the i8* used by the first one
    //remove that one's alloca and replace its uses by the last argument of the clone
    IntrinsicInst* va_start=0;
    IntrinsicInst* tmp=0;
    AllocaInst* alloca=0;
    BitCastInst* bitCast=0;
    LoadInst* load=0;
    bool firstStart=true;
    bool firstEnd=true;

    Function::arg_iterator last_arg=to->arg_begin();//replacement for vararg
    for(unsigned int k=1;k<to->arg_size();k++) last_arg++;

    for(Function::iterator block=to->begin();block!=to->end();block++){
      BasicBlock::iterator instruction_end=block->end();
      for(BasicBlock::iterator instruction=block->begin();instruction!=instruction_end;instruction++){
	if((tmp=dyn_cast<IntrinsicInst>(&*instruction))&&(tmp->getCalledFunction()->getName()==string("llvm.va_start"))){
	  if(firstStart){
	    va_start=tmp;//first one

	    //	    alloca=cast<AllocaInst>(tmp->getOperand(1));
	    bitCast=cast<BitCastInst>(tmp->getOperand(1));
	    alloca=cast<AllocaInst>(bitCast->getOperand(0));

	    Instruction* afterAlloca=getNextInstruction(alloca);
	    AllocaInst* newAlloca=new AllocaInst(last_arg->getType(),"",afterAlloca);
	    new StoreInst(last_arg,newAlloca,afterAlloca);

	    alloca->replaceAllUsesWith(newAlloca);
	    //	    alloca->replaceAllUsesWith(CastInst::createPointerCast(last_arg,alloca->getType(),"",alloca));
	    //	    alloca->replaceAllUsesWith(last_arg);
	    //	    tmp->setOperand(1,alloca);
	    //tmp->setOperand(1,last_arg);
	    bitCast->setOperand(0,alloca);

	    //put this va_start here in the delegating from-function AFTER these for-loops, but already move this alloca there
	    alloca->moveBefore(fakeTerminator);
	    bitCast->moveBefore(fakeTerminator);

	    //	    actuals.push_back(alloca);//add it as actual of to
	    //	    actuals.push_back(bitCast);//add it as actual of to
	    load=new LoadInst(alloca,"",fakeTerminator);
	    actuals.push_back(load);//add it as actual of to
	    firstStart=false;
	  }else{
	    tmp->eraseFromParent();//users are using Argument already!
	  }
	}else if((tmp=dyn_cast<IntrinsicInst>(&*instruction))&&(tmp->getCalledFunction()->getName()==string("llvm.va_end"))){
	  //look for first end corresponding to last_arg and put it at end of original Function (before terminating instruction); terminate other such ends (not those for other variables)
	  //=> solves problem with two va_ends
	  if(tmp->getOperand(1)==last_arg){
	    if(firstEnd){
	      va_end=tmp;//first one
	      //	      tmp->setOperand(1,alloca);
	      tmp->setOperand(1,bitCast);
	      firstEnd=false;
	    }else{
	      //		new CallInst(module->getNamedFunction(string("llvm.va_copy")))
	      tmp->eraseFromParent();//users are using Argument already!
	    }
	  }
	}
      }
    }

    if(firstStart) actuals.push_back(Constant::getNullValue(last_arg->getType()));
    else if(va_start) va_start->moveBefore(load);//do moving here instead of in inner loop to avoid problems when instruction++
  }
    
  //create call
  //  FunctionType* FTy=(FunctionType*)(to->getFunctionType());
  //  cout << "TEST: " << FTy->isVarArg() << " - " << actuals.size() << " - " << FTy->getNumParams() << endl;

  //  Value** actuals_array=new Value*[actuals.size()];
  //  for(unsigned int iter=0;iter<actuals.size();iter++) actuals_array[iter]=actuals[iter];
  CallInst* returnValue=new CallInst((Function*)to,actuals.begin(),actuals.end(),string(""),fakeTerminator);
  //  delete[] actuals_array;

  //add return
  ReturnInst* realTerminator=new ReturnInst(returnValue,fakeTerminator);

  fakeTerminator->eraseFromParent();

  if(varArg && va_end) va_end->moveBefore(realTerminator);//do it here instead of in inner loop to avoid problems when instruction++

  return returnValue;
}

Type* cloneAndChangeElementType(Type* t,const map<Type*,Type*>& typeMap){
  //1. look if t is contained in typeMap
  map<Type*,Type*>::const_iterator it=typeMap.find(t);
  if(it!=typeMap.end()) return it->second;
  else{
    //2. if not, check if it is a compound type
    PointerType* s=0;
    ArrayType* a=0;
    if((s=dyn_cast<PointerType>(t))){
      Type* newT=cloneAndChangeElementType((Type*)s->getElementType(),typeMap);
      if(newT==Type::VoidTy){
	return PointerType::get(Type::Int8Ty);
      }else return PointerType::get(newT);
    }else if((a=dyn_cast<ArrayType>(t))){
      return ArrayType::get(cloneAndChangeElementType((Type*)a->getElementType(),typeMap),a->getNumElements());
    }else{//don't support type variables in StructType
      //cout << "WARNING: StructType type parameters not supported yet..." << endl;
      //return 0;
      /*      Type* try_something=PointerType::get(PointerType::get(t));//hack for CSOM
      it=typeMap.find(try_something);
      if(it!=typeMap.end()) return it->second;
      else return t;*/
      return t;
    } 
  }
}
