//===-- Context.cpp - Implement the Context class -------===//
//
//                     The LLVM Compiler Infrastructure
//
// This file was developed by the LLVM research group.  It is distributed under 
// the University of Illinois Open Source License. See autoconf/LICENSE.TXT for details.
//
// [02/10/2006] Adapted for Aspicere2 by Bram Adams.
//
//===----------------------------------------------------------------------===//
//
// This file implements the Context class for Aspicere2.
//
//===----------------------------------------------------------------------===//

#include "matcher/Context.h"
#include "utility/Util.h"
#include "llvm/GlobalValue.h"
#include "llvm/ADT/StringExtras.h"
#include <algorithm>
#include <iostream>

using namespace llvm;

#define DEBUG_SYMBOL_TABLE 0
#define DEBUG_ABSTYPE 0

// Class destructor
Context::~Context() {
#ifndef NDEBUG   // Only do this in -g mode...
  bool LeftoverValues = true;
  for (v_iterator VI = vmap.begin(), VE = vmap.end(); VI != VE; ++VI)
    if (!isa<Constant>(VI->second.first) ) {
      std::cerr << "Value still in symbol table! Type = '"
		<< (displayable_type((Type*)VI->second.first->getType())?VI->second.first->getType()->getDescription():"<too complex to print out>")
		<< "' Name = '"
                << VI->first << "'\n";
      LeftoverValues = false;
    }
  //  assert(LeftoverValues && "Values remain in symbol table!");

  bool LeftoverTypes = true;
  for (t_iterator VI = tmap.begin(), VE = tmap.end(); VI != VE; ++VI)
      std::cerr << "Type still in symbol table! Type = '"
                << (displayable_type((Type*)VI->second.first)?VI->second.first->getDescription():"<too complex to print out>")
		<< "' Name = '"
                << VI->first << "'\n";
      LeftoverTypes = false;
      //  assert(LeftoverTypes && "Types remain in symbol table!");
#endif
}

int Context::index(const std::string &Name){
  bool found=false;
  int res=0;
  const_v_iterator VI = vmap.find(Name);

  if (VI != vmap.end()){                   // We found the symbol
    found=true;
    KindOfContext kind=VI->second.second;
    assert(((kind==Context::ACTUAL) || (kind==Context::ARGUMENT) || (kind==Context::CROSSFUNCTION)) && std::string(std::string("Variable ")+Name+std::string(" is no actual, argument or cross-function...")).c_str());
    const std::string n(Name);
    res=name_to_actual_or_argument_index[Name];
  }

  assert(found && std::string(std::string("Variable ")+Name+std::string(" is no actual, argument or cross-function...")).c_str());
  return res;
}

// getUniqueName - Given a base name, return a string that is either equal to
// it (or derived from it) that does not already occur in the symbol table for
// the specified type.
//
std::string Context::getUniqueName(const std::string &BaseName) const {
  std::string TryName = BaseName;
  const_v_iterator End = vmap.end();
  const_t_iterator TEnd = tmap.end();

  // See if the name exists
  while ((vmap.find(TryName) != End)||(tmap.find(TryName) != TEnd))            // Loop until we find a free
    TryName = BaseName + utostr(++LastUnique); // name in the symbol table
  return TryName;
}


// lookup a value - Returns null on failure...
//
Value *Context::lookupValue(const std::string &Name) const {
  const_v_iterator VI = vmap.find(Name);
  if (VI != vmap.end())                   // We found the symbol
    return const_cast<Value*>(VI->second.first);
  return 0;
}

//returns garbage value if not registered...
Context::KindOfContext Context::lookupKind(const std::string &Name) const{
  const_v_iterator VI = vmap.find(Name);
  if (VI != vmap.end())                   // We found the symbol
    return VI->second.second;

  const_t_iterator TI = tmap.find(Name);
  if (TI != tmap.end())                   // We found the symbol
    return TI->second.second;

  return CONSTANT;//garbage value
}

// lookup a type - Returns null on failure...
//
Type *Context::lookupType(const std::string &Name) const {
  const_t_iterator VI = tmap.find(Name);
  if (VI != tmap.end())                   // We found the symbol
    return const_cast<Type*>(VI->second.first);
  return 0;
}

// Strip the symbol table of its names.
//
bool Context::strip() {
  bool RemovedSymbol = false;
  for (v_iterator VI = vmap.begin(), VE = vmap.end(); VI != VE; ) {
    Value *V = VI->second.first;
    ++VI;
    if (!isa<GlobalValue>(V) || cast<GlobalValue>(V)->hasInternalLinkage()) {
      // Set name to "", removing from symbol table!
      V->setName("");
      RemovedSymbol = true;
    }
  }
  /*  for (t_iterator VI = tmap.begin(), VE = tmap.end(); VI != VE; ) {
    Type *V = VI->second;
    ++VI;
    //    if (!isa<GlobalValue>(V) || cast<GlobalValue>(V)->hasInternalLinkage()) {
      // Set name to "", removing from symbol table!
      V->setName("");
      RemovedSymbol = true;
      //    }
      }*/
  return RemovedSymbol;
}

// Insert a value into the symbol table with the specified name...
//
void Context::insert(const std::string& name,Value* V,KindOfContext kind,int index_of_actual_or_argument,llvm::Value* stack) {
  assert(V && "Can't insert null Value into symbol table!");
  //  assert(V->hasName() && "Can't insert nameless Value into symbol table");

  // Check to see if there is a naming conflict.  If so, rename this type!
  std::string UniqueName = getUniqueName(name);

#if DEBUG_SYMBOL_TABLE
  dump();
  std::cerr << " Inserting value: " << UniqueName << ": " << V->dump() << "\n";
#endif

  // Insert the vmap entry
  vmap.insert(std::make_pair(UniqueName, std::make_pair(V,kind)));

  if((kind==Context::ACTUAL)||(kind==Context::ARGUMENT)||(kind==Context::CROSSFUNCTION)){
      name_to_actual_or_argument_index[name]=index_of_actual_or_argument;
  }

  if(kind==Context::CROSSFUNCTION){
    name_to_stack[name]=stack;
  }
}

// Insert a type into the symbol table with the specified name...
//
void Context::insert(const std::string& name,Type* T,KindOfContext kind) {
  assert(T && "Can't insert null Type into symbol table!");
  //  assert(V->hasName() && "Can't insert nameless Value into symbol table");

  // Check to see if there is a naming conflict.  If so, rename this type!
  std::string UniqueName = getUniqueName(name);

#if DEBUG_SYMBOL_TABLE
  dump();
  std::cerr << " Inserting type: " << UniqueName << ": " << T->dump() << "\n";
#endif

  // Insert the tmap entry
  tmap.insert(std::make_pair(UniqueName, std::make_pair(T,kind)));
}

// Remove a value
bool Context::erase(const std::string& name,Value *V) {
  //  assert(V->hasName() && "Value doesn't have name!");
  v_iterator Entry = vmap.find(name);
  if (Entry == vmap.end())
    return false;

#if DEBUG_SYMBOL_TABLE
  dump();
  std::cerr << " Removing Value: " << name << "\n";
#endif

  // Remove the value from the plane...
  vmap.erase(Entry);

  std::map<std::string,int>::iterator EEntry = name_to_actual_or_argument_index.find(name);
  if (EEntry != name_to_actual_or_argument_index.end()) name_to_actual_or_argument_index.erase(EEntry);

  std::map<std::string,llvm::Value*>::iterator SEntry = name_to_stack.find(name);
  if (SEntry != name_to_stack.end()) name_to_stack.erase(SEntry);

  return true;
}

// Remove a type
bool Context::erase(const std::string& name,Type *V) {
  //  assert(V->hasName() && "Value doesn't have name!");
  t_iterator Entry = tmap.find(name);
  if (Entry == tmap.end())
    return false;

#if DEBUG_SYMBOL_TABLE
  dump();
  std::cerr << " Removing Type: " << name << "\n";
#endif

  // Remove the type from the plane...
  tmap.erase(Entry);
  return true;
}


// rename - Given a value with a non-empty name, remove its existing entry
// from the symbol table and insert a new one for Name.  This is equivalent to
// doing "remove(V), V->Name = Name, insert(V)", 
//
bool Context::rename(const std::string& vName,Value *V, const std::string &name) {
  assert(V && "Can't rename a null Value");
  //  assert(V->hasName() && "Can't rename a nameless Value");
  //  assert(!V->getName().empty() && "Can't rename an Value with null name");
  //  assert(V->getName() != name && "Can't rename a Value with same name");
  assert(!name.empty() && "Can't rename a named Value with a null name");

  // Find the name
  v_iterator VI = vmap.find(vName);

  // If we didn't find it, we're done
  if (VI == vmap.end())
    return false;

  KindOfContext kind=VI->second.second;

  // Remove the old entry.
  vmap.erase(VI);

  // See if we can insert the new name.
  VI = vmap.lower_bound(name);

  // Is there a naming conflict?
  if (VI != vmap.end() && VI->first == name) {
    //V->Name = getUniqueName( name);
    //vmap.insert(make_pair(V->Name, V));
    std::string uniqueName=getUniqueName( name);
    vmap.insert(std::make_pair(uniqueName, std::make_pair(V,kind)));
    
    std::map<std::string,int>::iterator EEntry = name_to_actual_or_argument_index.find(vName);
    if (EEntry != name_to_actual_or_argument_index.end()){
      name_to_actual_or_argument_index[uniqueName]=EEntry->second;
      name_to_actual_or_argument_index.erase(EEntry);
    }

    std::map<std::string,llvm::Value*>::iterator SEntry = name_to_stack.find(vName);
    if (SEntry != name_to_stack.end()){
      name_to_stack[uniqueName]=SEntry->second;
      name_to_stack.erase(SEntry);
    }
  } else {
    //    V->Name = name;
    vmap.insert(VI, std::make_pair(name, std::make_pair(V,kind)));

    std::map<std::string,int>::iterator EEntry = name_to_actual_or_argument_index.find(vName);
    if (EEntry != name_to_actual_or_argument_index.end()){
      name_to_actual_or_argument_index[name]=EEntry->second;
      name_to_actual_or_argument_index.erase(EEntry);
    }

    std::map<std::string,llvm::Value*>::iterator SEntry = name_to_stack.find(vName);
    if (SEntry != name_to_stack.end()){
      name_to_stack[name]=SEntry->second;
      name_to_stack.erase(SEntry);
    }
  }

  return true;
  }

// rename - Given a type with a non-empty name, remove its existing entry
// from the symbol table and insert a new one for Name.  This is equivalent to
// doing "remove(T), T->Name = Name, insert(T)", 
//
bool Context::rename(const std::string& tName,Type *T, const std::string &name) {
  assert(T && "Can't rename a null Type");
  //  assert(V->hasName() && "Can't rename a nameless Value");
  //  assert(!V->getName().empty() && "Can't rename an Value with null name");
  //  assert(V->getName() != name && "Can't rename a Value with same name");
  assert(!name.empty() && "Can't rename a named Type with a null name");

  // Find the name
  t_iterator VI = tmap.find(tName);

  // If we didn't find it, we're done
  if (VI == tmap.end())
    return false;

  KindOfContext kind=VI->second.second;

  // Remove the old entry.
  tmap.erase(VI);

  // See if we can insert the new name.
  VI = tmap.lower_bound(name);

  // Is there a naming conflict?
  if (VI != tmap.end() && VI->first == name) {
    //V->Name = getUniqueName( name);
    //vmap.insert(make_pair(V->Name, V));
    tmap.insert(std::make_pair(getUniqueName( name), std::make_pair(T,kind)));
  } else {
    //    V->Name = name;
    tmap.insert(VI, std::make_pair(name, std::make_pair(T,kind)));
  }

  return true;
}

void Context::addStack(const std::string& vName,llvm::Value* stack){
  name_to_stack[vName]=stack;
}

llvm::Value* Context::getStack(const std::string& vName) const{
  std::map<const std::string,llvm::Value*>::const_iterator it=name_to_stack.find(vName);
  if(it!=name_to_stack.end()) return it->second;
  else return 0;
}

// DumpVal - a std::for_each function for dumping a value
//
static void DumpVal(const std::pair<const std::string, std::pair<Value *,Context::KindOfContext> > &V) {
  std::cerr << "  '" << V.first << "' = ";
  V.second.first->dump();
  std::cerr << " of kind " << V.second.second << "\n";
}

// DumpTyp - a std::for_each function for dumping a type
//
static void DumpTyp(const std::pair<const std::string, std::pair<Type *,Context::KindOfContext> > &T) {
  std::cerr << "  '" << T.first << "' = ";
  T.second.first->dump();
  std::cerr << " of kind " << T.second.second << "\n";
}

// dump - print out the symbol table
//
void Context::dump() const {
  std::cerr << "Context:\n";
  for_each(vmap.begin(), vmap.end(), DumpVal);
  for_each(tmap.begin(), tmap.end(), DumpTyp);
}
