/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Aspicere2.
 *
 * The Initial Developer of the Original Code is
 * the Ghislain Hoffman Software Engineering Lab, INTEC, University Ghent.
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Bram Adams <bram_DOT_adams_AT_ugent_DOT_be>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "matcher/JoinPointShadowSorter.h"
#include <iostream>
#include <set>

using namespace llvm;
using namespace std;

/**
 * Sort matches first by shadow (shadows are kept in incoming order, so before/after/... remains the same; order doesn't matter for weaving of introductions, only for matching, so they are all put up front) and then for each shadow by advice (1. intro 2. continuation 3. others). The latter is easy, as the matches are already sorted by advice in the argument.
 * @param matches all matches sorted by advice
 */
void JoinPointShadowSorter::sort(std::vector<Director*>* matches) const{
  multimap<llvm::Value*,Director*> introShadowsToMatches;
  multimap<llvm::Value*,Director*> continuationShadowsToMatches;
  multimap<llvm::Value*,Director*> otherShadowsToMatches;

  vector<llvm::Value*> intros_vector;
  vector<llvm::Value*> continuations_vector;
  vector<llvm::Value*> others_vector;

  set<llvm::Value*> intros_set;
  set<llvm::Value*> continuations_set;
  set<llvm::Value*> others_set;

  //DEBUG
  //  int nr_of_intros=0;
  //  int nr_of_continuations=0;
  //  int nr_of_others=0;
  //DEBUG

  for(std::vector<Director*>::iterator it=matches->begin();it!=matches->end();it++){
    JoinPoint* jp=(*it)->getJoinPoint();
    Value* key=jp->getShadow();
    string jp_type(jp->getType());
    string jp_time(jp->getTime());
    //    cout << "[SORTER] " << jp_type << endl;

    if(jp_time=="intro"){
      introShadowsToMatches.insert(introShadowsToMatches.upper_bound(key), multimap<llvm::Value*,Director*>::value_type(key, *it));//make sure values stay ordered too    
      if(intros_set.insert(key).second) intros_vector.push_back(key);
      //      nr_of_intros++;
    }else if(jp_type=="continuation"){
      continuationShadowsToMatches.insert(continuationShadowsToMatches.upper_bound(key), multimap<llvm::Value*,Director*>::value_type(key, *it));//make sure values stay ordered too    
      if(continuations_set.insert(key).second) continuations_vector.push_back(key);
      //      nr_of_continuations++;
    }else{
      otherShadowsToMatches.insert(otherShadowsToMatches.upper_bound(key), multimap<llvm::Value*,Director*>::value_type(key, *it));//make sure values stay ordered too    
      if(others_set.insert(key).second) others_vector.push_back(key);
      //      nr_of_others++;
    }
  }

  matches->clear();

  //first treat all shadows for which intros exist
  //  for(multimap<llvm::Value*,Director*>::iterator it=introShadowsToMatches.begin();it!=introShadowsToMatches.end();it++){
  //    shadows.insert(it->first);
  //  }

  for(vector<llvm::Value*>::iterator it=intros_vector.begin();it!=intros_vector.end();it++){
    pair<multimap<llvm::Value*,Director*>::iterator,multimap<llvm::Value*,Director*>::iterator> intros=introShadowsToMatches.equal_range(*it);
    for(multimap<llvm::Value*,Director*>::iterator itt=intros.first;itt!=intros.second;itt++){
      matches->push_back(itt->second);
      //      nr_of_intros--;
    }

    pair<multimap<llvm::Value*,Director*>::iterator,multimap<llvm::Value*,Director*>::iterator> continuations=continuationShadowsToMatches.equal_range(*it);
    for(multimap<llvm::Value*,Director*>::iterator itt=continuations.first;itt!=continuations.second;itt++){
      matches->push_back(itt->second);
    }
    continuationShadowsToMatches.erase(continuations.first,continuations.second);

    pair<multimap<llvm::Value*,Director*>::iterator,multimap<llvm::Value*,Director*>::iterator> others=otherShadowsToMatches.equal_range(*it);
    for(multimap<llvm::Value*,Director*>::iterator itt=others.first;itt!=others.second;itt++){
      matches->push_back(itt->second);
    }
    otherShadowsToMatches.erase(others.first,others.second);
  }

  //now treat all shadows for which continuations exist, but no intros
  //  shadows.clear();
  set<llvm::Value*> shadows(intros_set.begin(),intros_set.end());//will contain all shadows seen
  //  for(multimap<llvm::Value*,Director*>::iterator it=continuationShadowsToMatches.begin();it!=continuationShadowsToMatches.end();it++){
  //    shadows.insert(it->first);
  //  }

  for(vector<llvm::Value*>::iterator it=continuations_vector.begin();it!=continuations_vector.end();it++){
    if(shadows.find(*it)!=shadows.end()) continue;//ignore previously seen shadows
    else shadows.insert(*it);

    pair<multimap<llvm::Value*,Director*>::iterator,multimap<llvm::Value*,Director*>::iterator> continuations=continuationShadowsToMatches.equal_range(*it);
    for(multimap<llvm::Value*,Director*>::iterator itt=continuations.first;itt!=continuations.second;itt++){
      matches->push_back(itt->second);
    }

    pair<multimap<llvm::Value*,Director*>::iterator,multimap<llvm::Value*,Director*>::iterator> others=otherShadowsToMatches.equal_range(*it);
    for(multimap<llvm::Value*,Director*>::iterator itt=others.first;itt!=others.second;itt++){
      matches->push_back(itt->second);
    }
    otherShadowsToMatches.erase(others.first,others.second);
  }

  //finally treat remaining shadows for which no continuations exist, nor intros
  //  shadows.clear();
  //  for(multimap<llvm::Value*,Director*>::iterator it=otherShadowsToMatches.begin();it!=otherShadowsToMatches.end();it++){
  //    shadows.insert(it->first);
  //  }

  for(vector<llvm::Value*>::iterator it=others_vector.begin();it!=others_vector.end();it++){
    if(shadows.find(*it)!=shadows.end()) continue;//ignore previously seen shadows

    pair<multimap<llvm::Value*,Director*>::iterator,multimap<llvm::Value*,Director*>::iterator> others=otherShadowsToMatches.equal_range(*it);
    for(multimap<llvm::Value*,Director*>::iterator itt=others.first;itt!=others.second;itt++){
      matches->push_back(itt->second);
    }
  }
}
