/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Aspicere2.
 *
 * The Initial Developer of the Original Code is
 * the Ghislain Hoffman Software Engineering Lab, INTEC, University Ghent.
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Bram Adams <bram_DOT_adams_AT_ugent_DOT_be>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */


/**
 * Interface to SWI-Prolog engine allowing the addition of facts and querying of rules.
 *
 * @author Bram Adams
 */

#define DEBUG_TYPE "matcher"

#include "matcher/Context.h"
#include "prolog/NativePublicAPI.h"
#include "prolog/QueryEngine.h"
#include "matcher/AfterJoinPoint.h"
#include "matcher/AroundJoinPoint.h"
#include "matcher/BeforeJoinPoint.h"
#include "matcher/IntroJoinPoint.h"
#include "matcher/CallDirector.h"
#include "matcher/ExecutionDirector.h"
#include "matcher/ContinuationDirector.h"
#include "llvm/Support/Debug.h"
#include <cassert>
#include <iostream>

using namespace llvm;
using namespace std;

namespace{
  void print(ostream& out,const std::string& functor,const std::vector<llvm::Value*>& ptr_args,const std::vector<llvm::Type*>& type_args,const std::vector<int>& int_args,const std::vector<std::string>& string_args){
    out << functor << "(";

    int nr_of_args=ptr_args.size()+type_args.size()+int_args.size()+string_args.size();
    int index=1;
    for(unsigned int i=0;i<ptr_args.size();i++){
      out << ptr_args[i];
      if((index++)<nr_of_args) out << ",";
    }
    for(unsigned int i=0;i<type_args.size();i++){
      out << type_args[i];
      if((index++)<nr_of_args) out << ",";
    }
    for(unsigned int i=0;i<int_args.size();i++){
      out << int_args[i];
      if((index++)<nr_of_args) out << ",";
    }
    for(unsigned int i=0;i<string_args.size();i++){
      out << "'" << string_args[i].c_str() << "'";
      if((index++)<nr_of_args) out << ",";
    }
    out << ")." << endl;
  }

  JoinPoint* getJoinPoint(const string& time,const std::string& aspect,const std::string advice,llvm::Value* shadow,const std::string& shadowType,vector<string>* variables,Context* context,JoinPoint::ResidueMap* residues,JoinPoint::ResidueMap* miscResidues,int is_dummy){
    JoinPoint* jp=0;

    if(time==string("after")){
      jp=new AfterJoinPoint(aspect,advice,shadow,shadowType,variables,context,false,residues,miscResidues);
    }else if(time==string("after_returning")){
      jp=new AfterJoinPoint(aspect,advice,shadow,shadowType,variables,context,true,residues,miscResidues);
    }else if(time==string("before")){
      jp=new BeforeJoinPoint(aspect,advice,shadow,shadowType,variables,context,residues,miscResidues);
    }else if(time==string("around")){
      jp=new AroundJoinPoint(aspect,advice,shadow,shadowType,variables,context,residues,miscResidues);
    }else if(time==string("intro")){
      jp=new IntroJoinPoint(aspect,advice,shadow,shadowType,variables,context,residues,miscResidues);
    }else return 0;

    jp->setDummy(is_dummy);
    return jp;
  }

  Director* getDirector(const string& type,JoinPoint* jp){
    if(type=="call"){
      return new CallDirector(jp);
    }else if(type=="execution"){
      return new ExecutionDirector(jp);
    }else if(type=="continuation"){
      return new ContinuationDirector(jp);
    }else return 0;
  }

  void print_swi_term_type(PlTerm& p){
    switch(p.type()){
      case PL_VARIABLE:
	  std::cout << "VARIABLE" << endl;
	  break;
      case PL_FLOAT:
	  std::cout << "FLOAT" << endl;
	  break;
      case PL_INTEGER:
	  std::cout << "INTEGER" << endl;
	  break;
      case PL_ATOM:
	  std::cout << "ATOM" << endl;
	  break;
      case PL_STRING:
	  std::cout << "STRING" << endl;
	  break;
      case PL_TERM:
	  std::cout << "TERM" << endl;
	  break;
      default:
	  std::cout << "ELSE" << endl;
	  break;
      }
  }

  // Code snippet of Alan Baljeu (alanb AT cornerstonemold.com), adapted by Bram Adams (eliminated Conv-argument)
  // begin and end -STL style iterators delimiting a collection of stuff which f can convert.
  // Conv f - is a method or function-class which takes a list element and yields a PlTerm.
  
  template <typename FwdIt, typename Conv>
  PlTerm ListToProlog(FwdIt begin, FwdIt end, Conv f)
  {
    PlTerm a;
    PlTail p(a);
    
    for ( ; begin != end; ++begin)
      {
	try {
	  p.append(f(*begin));
	}
	catch (PlException &e)
	  {
	    std::cout << "PROLOG EXCEPTION: " << e.operator char*() << std::endl;
	  }
      }
    
    p.close();
    return a;
  }

  template <typename FwdIt>
  PlTerm ListToProlog(FwdIt begin, FwdIt end)
  {
    PlTerm a;
    PlTail p(a);
    
    for ( ; begin != end; ++begin)
      {
	try {
	  p.append(*begin);
	}
	catch (PlException &e)
	  {
	    std::cout << "PROLOG EXCEPTION: " << e.operator char*() << std::endl;
	  }
      }
    
    p.close();
    return a;
  }

  // Helper method to alleviate problems when one function needs more than one query to perform: PlQuery's destructor needs to close things apparently...
  void do_query(const char* predicate,const PlTermv& args,const char* error_msg){
    PlQuery q(predicate, args);
    assert(q.next_solution() && error_msg );
  }

}

QueryEngine::QueryEngine() throw (QueryEngineException){
  char *plav[9];
  string x_switch("-x");
  string l_switch("-L128m");
  string g_switch("-G128m");
  string t_switch("-T128m");
  string a_switch("-A128m");
  string no_switch("-nosignals");

  //  plav[0] = "@PROLOG_STATE_NAME@";
  plav[0] = getenv("PROLOG_STATE_NAME");
  plav[1] = (char*)x_switch.c_str();
  //  plav[2] = "@PROLOG_STATE_PATH@";
  plav[2] = getenv("PROLOG_STATE_PATH");
  plav[3] = (char*)l_switch.c_str();
  plav[4] = (char*)g_switch.c_str();
  plav[5] = (char*)t_switch.c_str();
  plav[6] = (char*)a_switch.c_str();
  plav[7] = (char*)no_switch.c_str();
  plav[8] = NULL;
  
  engine=new PlEngine(8,plav);
  metaBase=new MetaFactBase;
  //  residues=new vector<pair<Instruction*,ResidueProcessor*> >;

  PlTermv fact_term(1);
  fact_term[0] = (void*)metaBase;
  do_query("init_meta_base",fact_term,"Initialising Prolog's MetaFactBase did not succeed....");
}

QueryEngine::~QueryEngine(){
  delete engine;//halts it
  delete metaBase;
  //  delete residues;
}

void QueryEngine::addCallGraph(CallGraph& callGraph){
  PlTermv fact_term(1);
  fact_term[0] = (void*)&callGraph;
  do_query("init_call_graph",fact_term,"Initialising Prolog's call graph did not succeed....");
}

void QueryEngine::insert(const std::string& functor,const std::vector<llvm::Value*>& ptr_args,const std::vector<llvm::Type*>& type_args,const std::vector<int>& int_args,const std::vector<std::string>& string_args){
  static std::string error_msg("Asserting reified fact did not succeed using ");
  PlFrame fr;
  PlTermv args_term(ptr_args.size()+type_args.size()+int_args.size()+string_args.size());
  PlTerm list_of_pointers=PlAtom("[]");
  PlTerm list_of_types=PlAtom("[]");
  //  PlTermv assert_types_of_pointers_term(ptr_args.size());
  //  PlTermv assert_types_of_types_term(type_args.size());
  //  vector<Value*> pointers;
  //  vector<Type*> types;

  DEBUG(print(std::cerr,functor,ptr_args,type_args,int_args,string_args));

  unsigned int index=0;
  for(unsigned int i=0;i<ptr_args.size();i++){
    //    PlTermv tmp(2);
    Value* ptr=ptr_args[i];

    /*    tmp[0]=PlAtom("value");
    tmp[1]=ptr;
    PlQuery q("assert_pointer",tmp);
    assert(q.next_solution() && (error_msg+string("Asserting of Value* did not succeed...")).c_str() );*/

    metaBase->addValuePointer(ptr);
    args_term[index++]=(void*)ptr;
  }
  for(unsigned int i=0;i<type_args.size();i++){
    //    PlTermv tmp(2);
    Type* type=type_args[i];

    /*    tmp[0]=PlAtom("type");
    tmp[1]=type;
    PlQuery q("assert_pointer",tmp);
    assert(q.next_solution() && (error_msg+string("Asserting of Type* did not succeed...")).c_str() );*/

    metaBase->addTypePointer(type);
    args_term[index++]=(void*)type;
  }
  for(unsigned int i=0;i<int_args.size();i++){
    args_term[index++]=int_args[i];
  }
  for(unsigned int i=0;i<string_args.size();i++){
    args_term[index++]=PlCodeList(string_args[i].c_str());
    //    args_term[index++]=PlCharList(string_args[i].c_str());
  }

  PlTermv assert_term(1);
  assert_term[0] = PlCompound(functor.c_str(), args_term);
  do_query("assert",assert_term,(error_msg+functor).c_str());
  //  PlQuery q("assert", assert_term);
  //  assert(q.next_solution() && (error_msg+functor).c_str() );

  /*  PlTermv assert_term2(2);
  assert_term2[0] = PlAtom("value");
  assert_term2[1] = ListToProlog(pointers.begin(),pointers.end());
  do_query("assert_all_pointers",assert_term2,(error_msg+string("Asserting of Values did not succeed...")).c_str());*/
  //  PlCall("assert_all_pointers", assert_term2);
  //assert(q2.next_solution() && (error_msg+string("Asserting of Values did not succeed...")).c_str() );

  /*  PlTermv assert_term3(2);
  assert_term3[0] = PlAtom("type");
  assert_term3[1] = ListToProlog(types.begin(),types.end());
  do_query("assert_all_pointers",assert_term3,(error_msg+string("Asserting of Types did not succeed...")).c_str());*/
  //  PlCall("assert_all_pointers", assert_term3);
  //assert(q3.next_solution() && (error_msg+string("Asserting of Types did not succeed...")).c_str() );
}

std::vector<Director*>* QueryEngine::query(){
  std::vector<Director*>* matches=new std::vector<Director*>;

  try{
    //    PlFrame fr;
    PlTermv args(3);//metadata_list, Variables_pointer and Context_pointer (GONE: Residues, MiscResidues and Dummy)
    
    //    DEBUG(PlCall("listing"));
    //    DEBUG(PlCall("print_all_reified_predicates"));
    //    DEBUG(PlCall("match_all"));
    //PlCall("match_all");
    //DEBUG(PlCall("print_all_reified_predicates"));
    PlCall("mytest");

    // only call match
    int nr_of_matches=0;
    PlQuery q("match", args);
    std::cout << "[QUERYENGINE] Starting real match..." << std::endl;
    while( q.next_solution() ){
      DEBUG(std::cout << "[QUERYENGINE] Found match " << ++nr_of_matches << endl);

      //metadata_list: [['Jp',[Joinpoint_ptr,'call']],['ASPICERE2_ASPECT','logging'],['ASPICERE2_ADVICE','logging_nonvoid'],['ASPICERE2_ADVICE_TYPE','around']]
      //WARNING: e=[1,2,3]=.(1,.(2,.(3,empty))) => access 2 as e[2][1] (the . is always at index 0)
      PlTerm metadata_list=args[0];
      DEBUG(print_swi_term_type(metadata_list));
      PlTail tail(metadata_list);
      PlTerm e;
      
      tail.next(e);//e=['Jp',[Joinpoint_ptr,'call']]
      PlTerm pointer_elem=e[2][1][1];
      PlTerm type_elem=e[2][1][2][1];
      Value* shadow=(Value*)((void*)pointer_elem);
      string jp_type(type_elem);
      
      tail.next(e);//e=['ASPICERE2_ASPECT','logging']
      string aspect(e[2][1]);
      
      tail.next(e);//e=['ASPICERE2_ADVICE','logging_nonvoid']
      string advice(e[2][1]);
      
      tail.next(e);//e=['ASPICERE2_ADVICE_TYPE','around']
      string when(e[2][1]);
      
      //pointer to vector<string> containing context
      vector<string>* variables=(vector<string>*)((void*)args[1]);

      //pointer to SymbolTable containing context
      Context* context=(Context*)((void*)args[2]);

      //pointer to residue map
      //      JoinPoint::ResidueMap* new_residues=(JoinPoint::ResidueMap*)((void*)args[3]);
      JoinPoint::ResidueMap* new_residues=0;

      //pointer to miscellaneous residue map
      //      JoinPoint::ResidueMap* misc_residues=(JoinPoint::ResidueMap*)((void*)args[4]);
      JoinPoint::ResidueMap* misc_residues=0;

      //integer indicating whether dummy or not (1 or 0)
      //int is_dummy=(int)args[5];
      int is_dummy=0;
      
      //pointer to SymbolTable containing residues (currently unused)
      //      Context* residues=(Context*)((void*)args[3]);
      
      matches->push_back(getDirector(jp_type,getJoinPoint(when,aspect,advice,shadow,jp_type,variables,context,new_residues,misc_residues,is_dummy)));
    }
  }catch ( PlException &ex ){
    std::cerr << "*** ERROR occurred during querying:\t" << (char *)ex << "\t***" << std::endl;
  }catch ( ... ){
    std::cerr << "*** Some ERROR occurred during querying!" << std::endl;
  }

  DEBUG(std::cout << "[QUERYENGINE] Found " << matches->size() << " matches." << endl);
  //  cout << *matches << endl;  

  make_residues(matches);

  return matches;

  //  int PlTerm::type()
  //    Yields the actual type of the term as PL_term_type(). Return values are PL_VARIABLE, PL_FLOAT, PL_INTEGER, PL_ATOM, PL_STRING or PL_TERM

  //PlCall(functor, PlTermv(file));
  //predicate_t pred = PL_predicate("check_advice", 1, "user");
  //term_t h0 = PL_new_term_refs(1);
  //PL_put_atom_chars(h0, e);
  //return PL_call_predicate(NULL, PL_Q_NORMAL, pred, h0);
}

std::multimap<Value*,pair<Value*,int> >* QueryEngine::getCrossAdviceContext(){
  std::multimap<Value*,pair<Value*,int> >* res=new std::multimap<Value*,pair<Value*,int> >;

  PlTermv args(3);
  PlQuery q("cross_advice_context", args);
  std::cout << "[QUERYENGINE] Collecting cross-advice context..." << std::endl;
  while( q.next_solution() ){
    Value* shadow=(Value*)((void*)args[0]);
    Value* context=(Value*)((void*)args[1]);
    int index=int(args[2]);
    DEBUG(std::cout << "[QUERYENGINE] " << shadow << " has " << context << " on index " << index << endl);
    res->insert(make_pair(shadow,make_pair(context,index)));
  }

  return res;
}


