/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Aspicere2.
 *
 * The Initial Developer of the Original Code is
 * the Ghislain Hoffman Software Engineering Lab, INTEC, University Ghent.
 * Portions created by the Initial Developer are Copyright (C) 2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Bram Adams <bram_DOT_adams_AT_ugent_DOT_be>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#define DEBUG_TYPE "yes"

#include "reifier/ReifyingVisitor.h"
#include "stripdebug/DebugAnnotation.h"
#include "utility/Naming.h"
#include "llvm/Support/Casting.h"
#include "llvm/Support/Debug.h"
#include "llvm/DerivedTypes.h"
#include "llvm/Type.h"
#include <iostream>

using namespace std;
using namespace llvm;

namespace {
  //bool processing_aspects=false;
  bool no_function_pointers_found_yet=true;

  bool inAspect(const std::string& moduleName){
    return moduleName.find(".ac",0) != string::npos;
  }

  bool isSupportFunction(Function* fun){
    return fun->getName().find("__ASPICERE2_",0) != string::npos;
  }

  void ReifyingVisitor::visitFunction(llvm::Function &F){//execution
    //    F.viewCFG();
    //if(processing_aspects) return;
    DebugAnnotation* d=dynamic_cast<DebugAnnotation*>(F.getAnnotation(DebugAnnotation::debugID));
    if((d&&inAspect(d->getFileName()))||isSupportFunction(&F)) return;//d==0 e.g. for stdlib functions

    string fName=F.getName();

    DEBUG(if(fName.find("llvm.dbg.",0) != string::npos) return);

	  /*if(fName.find("ASPICERE2_",0)==0){
      processing_aspects=true;
      return;//currently no advice on advice
      }*/

    // methodDef(id,name)
    vector<llvm::Value*> ptr_args(1);
    vector<llvm::Type*> type_args(0);
    vector<int> int_args(0);
    vector<string> string_args(1);

    ptr_args[0]=&F;
    string_args[0]=fName;

    engine->insert(string("methodDecl"),ptr_args,type_args,int_args,string_args);
    if(F.begin()!=F.end()){//definition
      engine->insert(string("methodDef"),ptr_args,type_args,int_args,string_args);
    }

    // executionShadow(id)  
    string_args.resize(0);

    // remove this, as it's identical to methodDef and takes extra memory!
    //    engine->insert(string("executionShadow"),ptr_args,type_args,int_args,string_args);

    // methodType(id,type)

    ptr_args[0]=&F;
    type_args.resize(1);
    type_args[0]=(Type*)F.getReturnType();
    //    string_args.resize(0);
    //    string_args[0]=fName;
    //    string_args[1]=F.getReturnType();//->getDescription();

    //cout << "$$$ " << fName << ":\t" << string_args[0] << " _ " << string_args[1] << endl;
    engine->insert(string("methodType"),ptr_args,type_args,int_args,string_args);

    // methodArg(id,methoddef/decl_id,type,pos,nextpos,name)
    ptr_args.resize(2);
    int_args.resize(2);
    string_args.resize(1);

    int arg_counter=0;
    int nr_of_args_minus_one=F.arg_size()-1;
    //cout << "Method " << fName << ":\t\t";
    int index=0;
    int len=1;
    for(Function::arg_iterator it=F.arg_begin();it!=F.arg_end();it++){
      ptr_args[0]=it;
      ptr_args[1]=&F;
      type_args[0]=(Type*)it->getType();
      int_args[0]=arg_counter;
      if(arg_counter==nr_of_args_minus_one){
	int_args[1]=-1;
	//	cerr << "last argg: " << arg_counter << "/" << nr_of_args_minus_one << endl;
      }else{
	//	cerr << "argg: " << arg_counter << "/" << nr_of_args_minus_one << endl;
	int_args[1]=++arg_counter;
      }
      //      string_args[0]=it->getName();
      string_args[0]=get_name(it,len);
      //      string_args[1]=it->getType()->getDescription();
      //cout << "***" << ptr_args[0] << " _ " << ptr_args[1] << " _ " << int_args[0] << " _ " << int_args[1] << " _ " << string_args[0] << " _ " << string_args[1] << endl;
      //cout << "* ";
      engine->insert(string("methodArg"),ptr_args,type_args,int_args,string_args);

      index++;
      if(index==10) len=2;
    }

    //    cout << endl;

  }

  void ReifyingVisitor::visitModule(llvm::Module &M){//global vars
    // varDecl(id,type,name)
    vector<llvm::Value*> ptr_args(1);
    vector<llvm::Type*> type_args(1);
    vector<int> int_args(0);
    vector<string> string_args(1);

    for(Module::global_iterator it=M.global_begin();it!=M.global_end();it++){
      ptr_args[0]=cast<Value>(it);
      type_args[0]=(Type*)it->getType();
      string_args[0]=it->getName();
      //      string_args[1]=it->getType()->getElementType()->getDescription();
      //cout << "XXX " << string_args[0] << " _ " << string_args[1] << endl;
      bool ok=true;
      DEBUG(if(string_args[0].find("llvm.dbg.",0) != string::npos) ok=false);//continue would not continue this for-loop but DEBUG's do-while loop, hence the strange ok-construct
      if(ok) engine->insert(string("varDecl"),ptr_args,type_args,int_args,string_args);
    }
  }

  void ReifyingVisitor::visitCallInst(llvm::CallInst &I){//call
    //    if(processing_aspects) return;//currently no advice on advice
    llvm::Function* encloser=cast<Function>(I.getParent()->getParent());
    DebugAnnotation* d=dynamic_cast<DebugAnnotation*>(encloser->getAnnotation(DebugAnnotation::debugID));
    if((d&&inAspect(d->getFileName()))||isSupportFunction(encloser)) return;//d==0 e.g. for stdlib functions

    Function* F=I.getCalledFunction();
    DEBUG(if(F && F->getName().find("llvm.dbg.",0) != string::npos) return);

    // callShadow(id,methoddef/decl_id,enclosing_shadow_id)
    vector<llvm::Value*> ptr_args(3);
    vector<llvm::Type*> type_args(0);
    vector<int> int_args(0);
    vector<string> string_args(0);

    ptr_args[0]=&I;
    ptr_args[1]=F;//if == 0: function pointer
    ptr_args[2]=encloser;

    if(!F && no_function_pointers_found_yet){
      std::cout << "!!! FUNCTION POINTERS !!!" << endl;
      no_function_pointers_found_yet=false;
    }
    //    cout << "$$$ " << ((F)?F->getName():"0-ptr") << ":\t" << ptr_args[0] << " _ " << ptr_args[1] << " _ " << ptr_args[2] << endl;
    engine->insert(string("callShadow"),ptr_args,type_args,int_args,string_args);

    // callActual(id,call_id,pos,nextpos)
    ptr_args.resize(2);
    int_args.resize(2);

    int arg_counter=0;
    int nr_of_args_minus_one=I.getNumOperands()-1;
    //    cout << "Call to method " << F->getName() << ":\t\t";
    CallInst::op_iterator it=I.op_begin();
    it++;//get past function name
    for(;it!=I.op_end();it++){
      ptr_args[0]=it->get();//cast<Value>(it);
      ptr_args[1]=&I;
      int_args[0]=arg_counter;
      if(arg_counter==nr_of_args_minus_one){
	int_args[1]=-1;
	//	cerr << "last arg: " << arg_counter << "/" << nr_of_args_minus_one << endl;
      }else{
	//	cerr << "arg: " << arg_counter << "/" << nr_of_args_minus_one << endl;
	int_args[1]=++arg_counter;
      }

      //cout << "***" << ptr_args[0] << " _ " << ptr_args[1] << " _ " << int_args[0] << " _ " << int_args[1] << endl;
      //      cout << "* ";
      engine->insert(string("callActual"),ptr_args,type_args,int_args,string_args);
    }

    //    cout << endl;
  }

  void ReifyingVisitor::visitInvokeInst(llvm::InvokeInst &I){//call
    //    if(processing_aspects) return;//currently no advice on advice
    llvm::Function* encloser=cast<Function>(I.getParent()->getParent());
    DebugAnnotation* d=dynamic_cast<DebugAnnotation*>(encloser->getAnnotation(DebugAnnotation::debugID));
    if((d&&inAspect(d->getFileName()))||isSupportFunction(encloser)) return;//d==0 e.g. for stdlib functions

    Function* F=I.getCalledFunction();
    DEBUG(if(F && F->getName().find("llvm.dbg.",0) != string::npos) return);

    // callShadow(id,methoddef/decl_id,enclosing_shadow_id)
    vector<llvm::Value*> ptr_args(3);
    vector<llvm::Type*> type_args(0);
    vector<int> int_args(0);
    vector<string> string_args(0);

    ptr_args[0]=&I;
    ptr_args[1]=F;//if == 0: function pointer
    ptr_args[2]=encloser;

    if(!F && no_function_pointers_found_yet){
      std::cout << "!!! FUNCTION POINTERS !!!" << endl;
      no_function_pointers_found_yet=false;
    }
    engine->insert(string("callShadow"),ptr_args,type_args,int_args,string_args);

    // callActual(id,call_id,pos,nextpos)
    ptr_args.resize(2);
    int_args.resize(2);

    int arg_counter=0;
    int nr_of_args_minus_one=I.getNumOperands()-1;

    InvokeInst::op_iterator it=I.op_begin();
    it++;//get past function name
    for(;it!=I.op_end();it++){
      ptr_args[0]=cast<Value>(it);
      ptr_args[1]=&I;
      int_args[0]=arg_counter;
      if(arg_counter==nr_of_args_minus_one) int_args[1]=-1;
      else int_args[1]=++arg_counter;

      engine->insert(string("callActual"),ptr_args,type_args,int_args,string_args);
    }
  }

  void ReifyingVisitor::visitVAArgInst(llvm::VAArgInst &I){//call
    static int methodDeclared=0;

    //    if(processing_aspects) return;//currently no advice on advice
    llvm::Function* encloser=cast<Function>(I.getParent()->getParent());
    DebugAnnotation* d=dynamic_cast<DebugAnnotation*>(encloser->getAnnotation(DebugAnnotation::debugID));
    if((d&&inAspect(d->getFileName()))||isSupportFunction(encloser)) return;//d==0 e.g. for stdlib functions

    if(!methodDeclared){
      methodDeclared=666999665;//we only need unique integer, no real address (there is no real va_arg method!) => only call to va_arg can be advised

      vector<llvm::Value*> ptr_args(1);
      vector<llvm::Type*> type_args(0);
      vector<int> int_args(0);
      vector<string> string_args(1);
      
      ptr_args[0]=(Function*)((void*)methodDeclared);
      string_args[0]="va_arg";

      engine->insert(string("methodDecl"),ptr_args,type_args,int_args,string_args);
    }

    Function* F=(Function*)((void*)methodDeclared);//I.getCalledFunction();
    DEBUG(if(F && F->getName().find("llvm.dbg.",0) != string::npos) return);

    // callShadow(id,methoddef/decl_id,enclosing_shadow_id)
    vector<llvm::Value*> ptr_args(3);
    vector<llvm::Type*> type_args(0);
    vector<int> int_args(0);
    vector<string> string_args(0);

    ptr_args[0]=&I;
    ptr_args[1]=F;//if == 0: function pointer
    ptr_args[2]=encloser;

    if(!F && no_function_pointers_found_yet){
      std::cout << "!!! FUNCTION POINTERS !!!" << endl;
      no_function_pointers_found_yet=false;
    }
    //    cout << "$$$ " << ((F)?F->getName():"0-ptr") << ":\t" << ptr_args[0] << " _ " << ptr_args[1] << " _ " << ptr_args[2] << endl;
    engine->insert(string("callShadow"),ptr_args,type_args,int_args,string_args);

    // callActual(id,call_id,pos,nextpos)
    ptr_args.resize(2);
    int_args.resize(2);

    int arg_counter=0;
    int nr_of_args_minus_one=I.getNumOperands()-1;
    //    cout << "Call to method " << F->getName() << ":\t\t";
    CallInst::op_iterator it=I.op_begin();
    //    it++;//get past function name
    for(;it!=I.op_end();it++){
      ptr_args[0]=it->get();//cast<Value>(it);
      ptr_args[1]=&I;
      int_args[0]=arg_counter;
      if(arg_counter==nr_of_args_minus_one){
	int_args[1]=-1;
	//	cerr << "last arg: " << arg_counter << "/" << nr_of_args_minus_one << endl;
      }else{
	//	cerr << "arg: " << arg_counter << "/" << nr_of_args_minus_one << endl;
	int_args[1]=++arg_counter;
      }

      //cout << "***" << ptr_args[0] << " _ " << ptr_args[1] << " _ " << int_args[0] << " _ " << int_args[1] << endl;
      //      cout << "* ";
      engine->insert(string("callActual"),ptr_args,type_args,int_args,string_args);
    }

    //    cout << endl;
  }

}
