/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Lillambi.
 *
 * The Initial Developer of the Original Code is
 * the Software Engineering Lab, INTEC, University Ghent.
 * Portions created by the Initial Developer are Copyright (C) 2004
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Kris De Schutter <kris.deschutter@ugent.be>
 *   Bram Adams <bram.adams@ugent.be>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

/***
	@descr General-purpose predicates which are guaranteed to be loaded in at start-up of weaving process.
	@author Kris De Schutter
	@date
*/

:- load_library('be.ugent.ftw.intec.sel.lillambi.agents.TUPrologSystemLibrary').
:- load_library('be.ugent.ftw.intec.sel.lillambi.agents.TUPrologStringLibrary').

require_all:-
    %print('require_all...'), nl,
	require_all([])
	.
require_all(Prev):-
	findall([V,L],(
		we_require(V,L),
		require_lib(V,L)
	),New),
    %print('  processed sofar: '), write(New), nl,
	\+ New is Prev,
	!,
    %print('  another round needed'), nl,
	require_all(New)
	.
require_all(_). %:- print('  done ?'), nl.%end 


/**
	@form require(PrologModule)
	@constraints
	@ground PrologModule
	@descr Lazy loading of local Prolog-module.
*/
require(PrologModule) :-
  available(PrologModule),
  !.

require(PrologModule) :-%local file has priority
  \+ available(PrologModule),
  %print('Consulting '),write(PrologModule),nl,  
  consult(PrologModule),
  assert(available(PrologModule)),
  !.

/**
	@form require_lib(EnvironmentVariableName,PrologModule)
	@constraints
        @ground EnvironmentVariableName
	@unrestricted PrologModule
	@descr Lazy loading of global Prolog-module, with location determined by an environment variable (set by <code>plcafe</code>).
*/
require_lib(EnvironmentVariableName,PrologModule) :-%PrologModule is relatief pad (zonder starting /)
  sys_variable(EnvironmentVariableName,Path),
  !,%if path is defined, follow it
  str_cat([Path,PrologModule],PrologModuleWithPath),
  % print('need '), write(PrologModuleWithPath), nl,
  (
    available(PrologModuleWithPath)
    %print('already available'), nl
  ;
      \+available(PrologModuleWithPath),
      str_cat([PrologModuleWithPath,'.pl'],PrologModuleWithPathAndExtension),
    %print('consult '), write(PrologModuleWithPathAndExtension), nl,
      consult(PrologModuleWithPathAndExtension),
    %print('consulted'), nl,
      assert(available(PrologModuleWithPath))),
  !.
  
require_lib(EnvironmentVariableName,PrologModule) :-
	write('REQUIRE_LIB on variable '),write(EnvironmentVariableName),write(' and module '),write(PrologModule),write(' did NOT work!!!'),nl.

available(tools).

/**
	@form member(Member,List)
	@constraints
        @ground Member
	@ground List
	@descr Is Member a member of List? <p><font color=\"Red\">Bug: When X is variable, only first element of list is returned!</font></p>
*/
/*member(X,[X|List]).
member(X,[_|List]) :- member(X,List).*/

/**
	@form append_all(ListOfLists, AllListsAppended)
	@constraints
        @ground ListOfLists
	@unrestricted AllListsAppended
	@descr Appends all lists in ListOfLists into AllListsAppended.
*/
append_all([],[]).
append_all([Head | Tail], FullyAppended) :-
  append_all(Tail, Appended),
  append(Head, Appended, FullyAppended).

/**
	@form no_empties(List, ListWithoutEmpties)
	@constraints
        @ground List
	@unrestricted ListWithoutEmpties
	@descr Remove all empty sublists in List.
*/
no_empties([],[]).
no_empties([A|As], Bs) :-
  A = [],
  no_empties(As, Bs).
no_empties([A|As],[A|Bs]) :-
  \+ A = [],
  no_empties(As, Bs).

/**
	@form write_all(List)
	@constraints
        @ground List
	@descr Write out all elements of List without spaces.
*/
write_all([]).
write_all([A|As]) :- write(A), write_all(As).


/**
	@form delete(Item,List,ListWithoutItem)
	@constraints
        @ground Item
        @ground List
        @unrestricted ListWithoutItem
	@descr Delete Item from List.
*/
delete(_, [], []).
delete(Item, [Item|Rest], NewRest) :-
   delete(Item, Rest, NewRest).	
delete(Item, [First|Rest], [First|NewRest]) :-
   \+ First = Item,
   delete(Item, Rest, NewRest).	

/**
	@form delete_list(ItemList,List,ListWithoutItemList)
	@constraints
        @ground ItemList
        @ground List
        @unrestricted ListWithoutItemList
	@descr Delete all items from ItemList from List.
*/
delete_list([], List, List).
delete_list([Item|List], OldList, NewList):-
  delete(Item, OldList, TempList),
  delete_list(List, TempList, NewList).

% ternary logic 
ternary(yes).
ternary(maybe).
ternary(no). 

/**
	@form or(TernaryBooleanA,TernaryBooleanB,TernaryBooleanRes)
	@constraints
        @ground TernaryBooleanA
        @ground TernaryBooleanB
        @unrestricted TernaryBooleanRes
	@descr Ternary or of TernaryBooleanA and TernaryBooleanB. <p><font color=\"Red\">Bug: Doesn't work correctly when TernaryBooleanA or TernaryBooleanB is a variable!</font></p>
*/
or(yes,T,yes):- ternary(T).
or(T, yes ,yes):- ternary(T), \+ T = yes.
or(maybe,T,maybe):-ternary(T), \+ T = yes.
or(no,maybe,maybe).
or(no,no,no).

/**
	@form and(TernaryBooleanA,TernaryBooleanB,TernaryBooleanRes)
	@constraints
        @ground TernaryBooleanA
        @ground TernaryBooleanB
        @unrestricted TernaryBooleanRes
	@descr Ternary and of TernaryBooleanA and TernaryBooleanB. <p><font color=\"Red\">Bug: Doesn't work correctly when TernaryBooleanA or TernaryBooleanB is a variable!</font></p>
*/
and(no,T,no):- ternary(T).
and(T, no ,no):- ternary(T), \+ T = no.
and(maybe,T,maybe):-ternary(T), \+ T = no.
and(yes,maybe,maybe).
and(yes,yes,yes).

/**
	@form not(TernaryBoolean,TernaryBooleanRes)
	@constraints
        @ground TernaryBoolean
        @unrestricted TernaryBooleanRes
	@descr Ternary not of TernaryBoolean. <p><font color=\"Red\">Bug: Doesn't work correctly when TernaryBoolean is a variable!</font></p>
*/
not(no,yes).
not(maybe,maybe).
not(yes,no).


