/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Lillambi.
 *
 * The Initial Developer of the Original Code is
 * the Software Engineering Lab, INTEC, University Ghent.
 * Portions created by the Initial Developer are Copyright (C) 2004
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Kris De Schutter <kris.deschutter@ugent.be>
 *   Bram Adams <bram.adams@ugent.be>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

/***
	@descr Extensions of the XML predicates. Cuts are placed wherever the xml structure is modified. While this is defensive programming, it also reflects the nature of these predicates.
	@author Kris De Schutter
	@date June 2004
*/
:- load_library('be.ugent.ftw.intec.sel.lillambi.agents.TUPrologDOMXMLLibrary').

/**
	@form xml_previous(Current, Previous)
	@constraints
        @ground Current
        @unrestricted Previous
	@descr Find Previous node for the Current node.
*/
xml_previous(Node, Previous) :-
  xml_next(Previous, Node).

/**
	@form xml_append(NodeToAppend, NodeAfterWhichToAppend)
	@constraints
        @ground NodeToAppend
        @ground NodeAfterWhichToAppend
	@descr Append NodeToAppend to NodeAfterWhichToAppend.
*/
xml_append(Node, After) :-
  xml_next(After, Next),
  xml_prepend(Node, Next),
  !.

xml_append(Node, After) :-
  xml_parent(After, Parent),
  xml_last_child(Parent, After),
  xml_append_child(Node, Parent),
  !.

/**
	@form xml_prepend_child(NewChild, Parent)
	@constraints
        @ground NewChild
        @ground Parent
	@descr Prepend NewChild before first child of Parent.
*/
xml_prepend_child(Child, Parent) :-
  xml_first_child(Parent, First),
  xml_prepend(Child, First),
  !.

xml_prepend_child(Child, Parent) :-
  xml_append_child(Child, Parent),
  !.

/**
	@form xml_append_all_children(NewChildrenList, Parent)
	@constraints
        @ground NewChildrenList
        @ground Parent
	@descr Append the NewChildrenList after the last child of Parent.
*/
xml_append_all_children([], _) :- !.
xml_append_all_children([Child|Others], Parent) :-
  xml_append_child(Child, Parent),
  xml_append_all_children(Others, Parent),
  !.

/**
	@form xml_prepend_all_children(NewChildrenList, Parent)
	@constraints
        @ground NewChildrenList
        @ground Parent
	@descr Prepend the NewChildrenList before the first child of Parent.
*/
xml_prepend_all_children([], _) :- !.
xml_prepend_all_children([Child|Others], Parent) :-
  xml_prepend_all_children(Others, Parent),
  xml_prepend_child(Child, Parent),
  !.

/**
	@form xml_append_all(NodeList, NodeAfterWhichToAppend)
	@constraints
        @ground NodeList
        @ground NodeAfterWhichToAppend
	@descr Append the NodeList after NodeAfterWhichToAppend.
*/
xml_append_all([], _) :- !.
xml_append_all([Node|Others], After) :-
  xml_append(Node, After),
  xml_append_all(Others, After),
  !.

/**
	@form xml_prepend_all(NodeList, NodeBeforeWhichToPrepend)
	@constraints
        @ground NodeList
        @ground NodeAfterWhichToAppend
	@descr Prepend the NodeList before NodeBeforeWhichToPrepend.
*/
xml_prepend_all([], _) :- !.
xml_prepend_all([Node|Others], Before) :-
  xml_prepend(Node, Before),
  xml_prepend_all(Others, Before),
  !.

/**
	@form xml_remove_all(NodeList)
	@constraints
        @ground NodeList
	@descr Remove all nodes in NodeList from their XML-tree.
*/
xml_remove_all([]) :- !.
xml_remove_all([Node|Others]) :-
  xml_remove_all(Others),
  xml_remove(Node),
  !.

/**
	@form xml_all_types(NodeList, TypeList)
	@constraints
        @ground NodeList
        @unrestricted TypeList
	@descr Gather type of every node in NodeList.
*/
xml_all_types([],[]).
xml_all_types([A|B],[X|Y]) :-
  xml_type(A,X),
  !,
  xml_all_types(B,Y).

/**
	@form xml_is_element(Node)
	@constraints
        @ground Node
	@descr Is Node an element node?
*/
xml_is_element(X) :- xml_type(X, 1).

/**
	@form xml_is_attribute(Node)
	@constraints
        @ground Node
	@descr Is Node an attribute node?
*/
xml_is_attribute(X) :- xml_type(X, 2).

/**
	@form xml_is_text(Node)
	@constraints
        @ground Node
	@descr Is Node a text node?
*/
xml_is_text(X) :- xml_type(X, 3).

/**
	@form xml_is_cdata_section(Node)
	@constraints
        @ground Node
	@descr Is Node a cdata_section node?
*/
xml_is_cdata_section(X) :- xml_type(X, 4).

/**
	@form xml_is_entity_reference(Node)
	@constraints
        @ground Node
	@descr Is Node an entity_reference node?
*/
xml_is_entity_reference(X) :- xml_type(X, 5).

/**
	@form xml_is_entity(Node)
	@constraints
        @ground Node
	@descr Is Node an entity node?
*/
xml_is_entity(X) :- xml_type(X, 6).

/**
	@form xml_is_processing_instruction(Node)
	@constraints
        @ground Node
	@descr Is Node a processing_instruction node?
*/
xml_is_processing_instruction(X) :- xml_type(X, 7).

/**
	@form xml_is_comment(Node)
	@constraints
        @ground Node
	@descr Is Node a comment node?
*/
xml_is_comment(X) :- xml_type(X, 8).

/**
	@form xml_is_document(Node)
	@constraints
        @ground Node
	@descr Is Node a document node?
*/
xml_is_document(X) :- xml_type(X, 9).

/**
	@form xml_is_document_type(Node)
	@constraints
        @ground Node
	@descr Is Node a document_type node?
*/
xml_is_document_type(X) :- xml_type(X, 10).

/**
	@form xml_is_document_fragment(Node)
	@constraints
        @ground Node
	@descr Is Node a document_fragment node?
*/
xml_is_document_fragment(X) :- xml_type(X, 11).

/**
	@form xml_is_notation(Node)
	@constraints
        @ground Node
	@descr Is Node a notation node?
*/
xml_is_notation(X) :- xml_type(X, 12).


/**
	@form xml_all_values(NodeList, ValueList)
	@constraints
        @ground NodeList
        @unrestricted ValueList
	@descr Gather value of every node in NodeList.
*/
xml_all_values([],[]).
xml_all_values([A|B],[X|Y]) :-
  xml_value(A,X),
  !,
  xml_all_values(B,Y).

/**
	@form xml_all_names(NodeList, NameList)
	@constraints
        @ground NodeList
        @unrestricted NameList
	@descr Gather name of every node in NodeList.
*/
xml_all_names([],[]).
xml_all_names([A|B],[X|Y]) :-
  xml_name(A,X),
  !,
  xml_all_names(B,Y).

/**
	@form xml_ancestor(Descendant, Ancestor)
	@constraints
        @ground Descendant
        @unrestricted Ancestor
	@descr Look for Ancestor of Descendant.
*/
xml_ancestor(X,Y) :-
  xml_parent(X,Y).
xml_ancestor(X,Y) :-
  xml_parent(X,A),
  xml_ancestor(A,Y).

/**
	@form xml_ancestors(Descendant, AncestorList)
	@constraints
        @ground Descendant
        @unrestricted Ancestor
	@descr Look for all ancestors of Descendant and put them into AncestorList in the right order.
*/
xml_ancestors(X,[Y|Z]) :-
  \+ xml_is_root(X),
  xml_parent(X,Y),
  !,
  xml_ancestors(Y,Z).
xml_ancestors(X,[]) :-
  xml_is_root(X).

/**
	@form xml_filter_elements(NodeList, FilteredNodeList)
	@constraints
        @ground NodeList
        @unrestricted FilteredNodeList
	@descr Put all element nodes of NodeList in FilteredNodeList.
*/
xml_filter_elements([],[]).
xml_filter_elements([A|B],[A|Y]) :-
  xml_is_element(A),
  !,
  xml_filter_elements(B,Y).
xml_filter_elements([A|B],Y) :-
  \+ xml_is_element(A),
  !,
  xml_filter_elements(B,Y).

/**
	@form xml_clone_all(NodeList, CloneList)
	@constraints
        @ground NodeList
        @unrestricted CLoneList
	@descr Clone NodeList into CloneList.
*/
xml_clone_all([], []) :- !.
xml_clone_all([Node|Others], [Clone|ClonedOthers]) :-
  xml_clone(Node, Clone),
  xml_clone_all(Others, ClonedOthers),
  !.
