/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Lillambi.
 *
 * The Initial Developer of the Original Code is
 * the Software Engineering Lab, INTEC, University Ghent.
 * Portions created by the Initial Developer are Copyright (C) 2004
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Kris De Schutter <kris.deschutter@ugent.be>
 *   Bram Adams <bram.adams@ugent.be>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

// Lillambi [many tongues // minya]
// (C) 2004 Kris DE SCHUTTER, SEL, INTEC, UGent

package be.ugent.ftw.intec.sel.lillambi;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

public final class Lillambi {

    public static final int NO_LOADER = -1;

    public static final int AGENT_ERROR = -2;

    public static final int NO_ROOT = -3;
    
    public static final int AGENT_CLASS_INIT_ERROR = -4;

    private static final String root;

    private static final String pathSeparator = File.separator;

    static {
        String tmp = System.getProperty("LILLAMBI");
        if (tmp == null) {
            System.err.println("System-variable LILLAMBI has not been set.");
            System.exit(NO_ROOT);
        }
        root = tmp.endsWith(pathSeparator) ? tmp : tmp + pathSeparator;
        /* System.out.println("Value of LILLAMBI-property: "+root); */
    }

    /** Holds factories. */
    private static HashMap factories = new HashMap();

    /** Holds agents. */
    private static HashMap registry = new HashMap();

    /** Holds (ordered!) user-provided agent locations, either absolute paths or relative to the directory 
     *  from where lillambi was invoked. */
    private static List locations = new ArrayList();

    public static void main(String[] args) {
        Agent loader = null;

        try {
            loader = find("lillambi/loader");
            loader.act(args);

        } catch (AgentNotFound e) {
            System.err.println("Lillambi failed to initialize.");
            System.err.println("[Stacktrace]");
            e.printStackTrace();
            System.exit(NO_LOADER);

        } catch (AgentNotLoaded e) {
            System.err.println("Lillambi failed to initialize.");
            System.err.println("[Stacktrace]");
            e.printStackTrace();
            System.exit(NO_LOADER);

        } catch (AgentException e) {
            /* System.err.println("Loader threw a tantrum: "); */
            System.err.println(e.getMessage ());
            System.err.println("[Stacktrace]");
            e.printStackTrace();
            System.exit(AGENT_ERROR);
        }
    }

    public static Agent find(final String agentID) throws AgentNotFound,
            AgentNotLoaded {
        if (!registry.containsKey(agentID)) {
            String pathToAgent = "";
            try {
                pathToAgent = findAgentOnSystem(agentID);
            } catch (AgentNotFound e) {
                throw new AgentNotFound("Could not find <" + agentID
                        + "> on system.", e);
            }

            String agentName = haveAgentName(agentID);
            Agent agent = loadAgent(pathToAgent, agentName);
            if (agent == null)
                throw new AgentNotLoaded("Could not load <" + agentName
                        + "> from <" + pathToAgent + ">.");

            registry.put(agentID, agent);
            return agent;
        } else {
            return (Agent) registry.get(agentID);
        }
    }

    private static Agent loadAgent(final String pathToAgent,
            final String agentName) throws AgentNotFound, AgentNotLoaded {
        Properties props = new Properties();
        try {
            props.load(new BufferedInputStream(new FileInputStream(pathToAgent
                    + agentName + ".properties")));

        } catch (FileNotFoundException e) {
            throw new AgentNotLoaded("Agent <" + agentName + "> not found at <"
                    + pathToAgent + agentName + ".properties>");

        } catch (IOException e) {
            throw new AgentNotLoaded(
                    "IO exception while reading properties for agent at <"
                            + pathToAgent + agentName + ".properties>");
        }

        String factory = "be.ugent.ftw.intec.sel.lillambi.agents."
                + props.getProperty("agent.factory");
        try {
            Class.forName(factory);

        } catch (ClassNotFoundException e) {
            throw new AgentNotLoaded("Could not load factory for defined in <"
                    + pathToAgent + agentName + ".properties>");
        }

        Factory theFactory = (Factory) factories.get(factory);
        props.setProperty("lillambi.agent_path", pathToAgent);
        props.setProperty("lillambi.agent_name", agentName);

        return theFactory.build(props);
    }

    private static String findAgentOnSystem(final String agent_identifier)
            throws AgentNotFound {
        Iterator it = locations.iterator();

        while (it.hasNext()) {
            String loc = (String) it.next();            
            File props = new File((loc.endsWith(pathSeparator) ? loc : loc
                    + pathSeparator)
                    + agent_identifier + ".properties");
            if (props.exists())
                return props.getParent() + pathSeparator;
        }
        File props = new File(root + "agents/" + agent_identifier
                + ".properties");
        if (props.exists())
            return props.getParent() + pathSeparator;
        else
            throw new AgentNotFound("");
    }

    private static String haveAgentName(final String agentID) {
        int last = agentID.lastIndexOf("/");
        String agentName = agentID.substring(last + 1);
        return agentName;
    }

    public static void addFactory(Factory factory) {
        factories.put(factory.getClass().getName(), factory);
    }

    /**
     * Get all extra agent locations.
     * @return Returns the locations.
     */
    public static List getLocations() {
        return locations;
    }

    /**
     * Set all extra agent locations.
     * @param locations The locations to set.
     */
    public static void setLocations(List locations) {
        Lillambi.locations = locations;
    }

    /**
     * Add new location to look for agents.
     * @param loc New agent location to add.
     */
    public static void addLocation(String loc) {
        Lillambi.locations.add(loc);
    }

    /**
     * Remove agent location.
     * @param loc Agent location to remove.
     */
    public static void removeLocation(String loc) {
        Lillambi.locations.remove(loc);
    }

    /**
     * @return Returns the root.
     */
    public static String getRoot() {
        return root;
    }
}

