/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Lillambi.
 *
 * The Initial Developer of the Original Code is
 * the Software Engineering Lab, INTEC, University Ghent.
 * Portions created by the Initial Developer are Copyright (C) 2004
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Kris De Schutter <kris.deschutter@ugent.be>
 *   Bram Adams <bram.adams@ugent.be>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package be.ugent.ftw.intec.sel.lillambi.agents;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.io.PrintWriter;

import be.ugent.ftw.intec.sel.lillambi.Agent;
import be.ugent.ftw.intec.sel.lillambi.AgentException;

public class SystemAgent implements Agent {
    private String command = null;

    private String[] parts = null;

    private int[] map = null;

    private int upper = -1;
    
    private String root = null;
    private String stdin = null;
    private int stdinIndex = -1;
    private String stdout = null;
    private int stdoutIndex = -1;
    private String stderr = null;
    private int stderrIndex = -1;

    SystemAgent (String root, String command, String stdin, String stdout, String stderr) {
        this.command = command;

	this.root = (root != null)? System.getProperty (root) : "";

	// Looking for parameter indexes...
        parts = command.split ("\\$[0-9]+", -1);
        map = new int[parts.length - 1];

        int start = 0;
        for (int i = 0; i < map.length; i++) {
            start += parts[i].length () + 1;
            int end = start + 1;
            while (end < command.length ()
                    && Character.isDigit (command.charAt (end)))
                end++;
            String substring = command.substring (start, end);
            map[i] = Integer.parseInt (substring);
            if (map[i] > upper)
                upper = map[i];
            start = end;
        }
        
        this.stdin = stdin;
        if (stdin != null && stdin.matches ("\\$[0-9]+"))
            stdinIndex = Integer.parseInt (stdin.substring (1));
        if (stdinIndex > upper)
            upper = stdinIndex;

        this.stdout = stdout;
        if (stdout != null && stdout.matches ("\\$[0-9]+"))
            stdoutIndex = Integer.parseInt (stdout.substring (1));
        if (stdoutIndex > upper)
            upper = stdoutIndex;

        this.stderr = stderr;
        if (stderr != null && stderr.matches ("\\$[0-9]+"))
            stderrIndex = Integer.parseInt (stderr.substring (1));        
        if (stderrIndex > upper)
            upper = stderrIndex;
    }

    public Object[] act (Object[] args) throws AgentException {
        if (args.length - 1 < upper)
            throw new AgentException ("Not enough parameters passed. Needed "
                    + (upper + 1) + ". Found only " + args.length + ".");

        String[] stringedArgs = new String[args.length];
        for (int i = 0; i < args.length; i++) {
            if (!(args[i] instanceof String))
                throw new AgentException ("Parameter " + i + " <" + args[i]
                        + "> is not a String.");
            stringedArgs[i] = (String) args[i];
        }

        String compiled = parts[0];
        for (int i = 0; i < map.length; i++) {
            compiled += stringedArgs[map[i]];
            compiled += parts[i + 1];
        }

        String in = stdin;
        if (stdinIndex != -1) in = stringedArgs[stdinIndex];
        String out = stdout;
        if (stdoutIndex != -1) out = stringedArgs[stdoutIndex];
        String err = stderr;
        if (stderrIndex != -1) err = stringedArgs[stderrIndex];
        
        try {
	    BufferedReader fileIn=null;
            if (stdinIndex != -1)
		fileIn=new BufferedReader(new FileReader(in));

            Process p = Runtime.getRuntime ().exec (root + compiled);
	    
            StreamGobbler outGobbler
                = new StreamGobbler (p.getInputStream (), out);
            StreamGobbler errGobbler
                = new StreamGobbler (p.getErrorStream (), err);
            outGobbler.start ();
            errGobbler.start ();

	    if (stdinIndex != -1) {
		String line = null;
		PrintStream pout = new PrintStream (new BufferedOutputStream (p.getOutputStream ()));
		while ((line = fileIn.readLine ()) != null) {
		    pout.println (line);
		}
		pout.close();
	    }

            int returnValue = p.waitFor ();

            return new Object[] {
                new Integer (returnValue),
                outGobbler.getGobbled (),
                errGobbler.getGobbled ()
                /*, out, err */
            };
	} catch (FileNotFoundException e){
	    throw new AgentException("Input file <"+in+"> for agent could not be found.",e);
        } catch (IOException e) {
            throw new AgentException (
                    "IOException while agent processed command <" + command
                            + ">.", e);

        } catch (InterruptedException e) {
            throw new AgentException ("Agent got rudely interrupted.", e);
        }
    }

    private class StreamGobbler implements Runnable {
        private String target;
        private InputStream is;
        private Thread thread;
        private String gobbled = "";
    
        public StreamGobbler (InputStream is, String target) {
            this.target = target;
            this.is = is;
        }
    
        public void start () {
            thread = new Thread (this);
            thread.start ();
        }
    
        public void run () {
            try {
                BufferedReader in
                    = new BufferedReader (new InputStreamReader (is));
                synchronized (this) { 
                    if (target == null) {
                        while (true) {
                            String s = in.readLine ();
                            if (s == null) break;
                            gobbled += s + "\n";
                        }
                    } else {
                        File file = new File (target);
                        PrintWriter writer
                            = new PrintWriter (
                                new BufferedWriter (
                                    new FileWriter (file, false)));

                        while (true) {
                            String s = in.readLine ();
                            if (s == null) {
                                writer.close ();
                                break;
                            }
                            writer.println (s);
                        }
                    }
                }
    
                in.close ();
    
            } catch (Exception ex) {
                System.err.println ("SystemAgent: problem reading stream:" + ex.getMessage ());
                ex.printStackTrace ();
            }
        }
        
        public synchronized String getGobbled () { return gobbled; }
    }
}

