#!/usr/bin/perl

# ***** BEGIN LICENSE BLOCK *****
# Version: MPL 1.1/GPL 2.0/LGPL 2.1
#
# The contents of this file are subject to the Mozilla Public License Version
# 1.1 (the "License"); you may not use this file except in compliance with
# the License. You may obtain a copy of the License at
# http://www.mozilla.org/MPL/
#
# Software distributed under the License is distributed on an "AS IS" basis,
# WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
# for the specific language governing rights and limitations under the
# License.
#
# The Original Code is Yerna Lindale.
#
# The Initial Developer of the Original Code is
# the Software Engineering Lab, INTEC, University Ghent.
# Portions created by the Initial Developer are Copyright (C) 2004
# the Initial Developer. All Rights Reserved.
#
# Contributor(s):
#   Kris De Schutter <kris.deschutter@ugent.be>
#
# Alternatively, the contents of this file may be used under the terms of
# either the GNU General Public License Version 2 or later (the "GPL"), or
# the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
# in which case the provisions of the GPL or the LGPL are applicable instead
# of those above. If you wish to allow use of your version of this file only
# under the terms of either the GPL or the LGPL, and not to allow others to
# use your version of this file under the terms of the MPL, indicate your
# decision by deleting the provisions above and replace them with the notice
# and other provisions required by the GPL or the LGPL. If you do not delete
# the provisions above, a recipient may use your version of this file under
# the terms of any one of the MPL, the GPL or the LGPL.
#
# ***** END LICENSE BLOCK *****

package PAL;
require LLL;
require PAL_Active_Rule_Finder;

# ===================================================================
# Main method...                                                    =
# ===================================================================

sub parse {
  my $fh_in = $_[0];

  my ($pal_file, $line_count, $line) = scan_directives ($fh_in, -1);
  ($line_count, my $tokens) = scan_grammar ($fh_in, $line, $line_count, $pal_file);
  parse_tokens ($tokens, $pal_file);
  $line_count += scan_epilogue ($fh_in, $pal_file);

  $pal_file->{'line_count'} = $line_count;

  return $pal_file;
}

# ===================================================================
# Scanning...                                                       =
# ===================================================================

sub scan_directives {
  my $fh_in = $_[0];
  my $line_count = $_[1];

  # This keeps track of tokens found...
  my $pal_file = PAL_File->new ();
  
  # Current line...
  my $line = '';

  # While end of directives has not been reached...
  while ($line = <$fh_in>) {

    # Bookkeeping...
    $line_count++;

    # Loose trailing newlines...
    chomp ($line);
  
    if ($line =~ /^\s*$/) {
      # Empty line.
      # Skip...
      next;
    }

    if ($line !~ /^#.*$/) {
      # No directive.
      # Exit...
      last;
    }

    if ($line =~ /^\#\@\s*([A-Za-z][A-Za-z0-9_]*)[^A-Za-z0-9_]?\s*$/) {
      # A naming directive.
      $pal_file->{'name'} = "$1";

    } elsif ($line =~ /^\#\>\s*([A-Za-z0-9._\/]+)\s*$/) {
      # An lll reference directive.
      $pal_file->{'lll_reference'} = "$1";

    } elsif ($line =~ /^\#\:\s*([A-Za-z][A-Za-z0-9_]*)[^A-Za-z0-9_]?\s*$/) {
      # A starter directive.
      $pal_file->{'starters'}->[@{$pal_file->{'starters'}}] = "$1";

    } else {
      scanner_error ("unrecognized directive at line $line_count\n");
    }
  }

  return $pal_file, $line_count, $line;
}

# ===================================================================

sub scan_grammar {
  my $fh_in = $_[0];
  my $line = $_[1];
  my $line_count = $_[2];
  my $pal_file = $_[3];

  my $end_of_file = 0;
  my $in_code_body = 0;
  my $in_prologdoc = 0;#added for PrologDoc-support: those docs are blindly copied to the .pl-file
  my $code_body = '';
  my $body_is_value = 0;
  my $tokens = [];

  # While end of file has not been reached...
  while ($end_of_file == 0) {
  
    # Read in new line if neccesary...
    if ($line =~ /^\s*$/) {
      if ($line = <$fh_in>) {
        # Bookkeeping...
        $line_count++;

        # Loose trailing newlines...
        chomp ($line);

        # If we reach the epilogue we stop...
        if ($line =~ /^\#\#\s*$/) { last; }

        # Start loop again...
        next;

      } else {
        # Flag end of file (not really necessary)...
        $end_of_file = 1;
        # Exit loop (this is why the previous is unnecessary)...
        last;
      }
    }
  
    if ($in_code_body == 1) {
      # We're processing a prolog body of code.
  
      # Skip whitespace...
      $line =~ /^\s*(.*)$/;
      $line = "$1";
  
      # Make sure newlines are added correctly...
      # (= add a newline unless it's empty)
      if (($code_body ne '') and ($line !~ /^\s*$/)) {
        $code_body = "$code_body\n";
      }
  
      # Look for the closing curly bracket...
      while (($in_code_body == 1) and ($line !~ /^\s*$/)) {
  
        # Match one of ", ' or }...
        if ($line =~ /^([^\"\'\}]*)(\"|\'|\})(.*)$/) {
  
          if ($2 eq '}') {
            # We found the closing curly bracket.
            # Flag end of body and gather up results...
            $in_code_body = 0;
            $code_body = "$code_body$1\n";
            $line = "$3";

            # DEBUG:
            # print STDERR ("CODE: $code_body\n");

            # For good measure...
            chomp ($code_body);

            # Build token...
            if ($body_is_value == 0) {
              $tokens->[@$tokens] = "{ $code_body }";
            } else {
              $tokens->[@$tokens] = "{= $code_body }";
            }
            $code_body = '';
            $body_is_value = 0;

          } else {
            # We found the start of a literal.
            # Gather up match so far...
            $code_body = "$code_body$1$2";
            my $delimiter = "$2";
            $line = "$3";

            # Scan for end of literal...
            my $scanning_literal = 1;
            while ($scanning_literal == 1) {

              if ($line =~ /^(\\.)(.*)/) {
                $code_body = "$code_body$1";
                $line = "$2";

              } elsif ($line =~ /^$delimiter(.*)/) {
                $code_body = "$code_body$delimiter";
                $line = "$1";
                $scanning_literal = 0;

              } elsif ($line =~ /^$/) {
                scanner_error ("incomplete literal at line $line_count\n");

              } else {
                $line =~ /^([^\\$delimiter]+)(.*)$/;
                $code_body = "$code_body$1";
                $line = "$2";
              }
            }
          } # End of matching ", ' or {.

        } else {
          # No closing curly brace.
          # Add current line to code body, and move on to next one...
            $code_body = "$code_body$line";
            $line = '';
        }
      } # End of looking for closing curly brace.
  
    } else {
      # Still processing lll specs.
  
      # Skip whitespace...
      $line =~ /^\s*(.*)$/;
      $line = "$1";
  
      while ($line !~ /^\s*$/) {
  
        # DEBUG:
        # print STDERR ("<$line>\n");
  
        # Tokenize the stream...
        if ($line =~ /^(\"|\')(.*)/) {
  
          # A string literal...
          my $literal = "";
          my $delimiter = "$1";
          $line = "$2";
  
          my $scanning_for_literal = 1;
          while ($scanning_for_literal == 1) {
  
            if ($line =~ /^(\\.)(.*)/) {
              $literal = "$literal$1";
              $line = "$2";
  
            } elsif ($line =~ /^$delimiter(.*)/) {
              $line = "$1";
              $scanning_for_literal = 0;
  
            } elsif ($line =~ /^$/) {
              scanner_error ("incomplete literal at line $line_count\n");
  
            } else {
              $line =~ /^([^\\$delimiter]+)(.*)$/;
              $literal = "$literal$1";
              $line = "$2";
            }
          }
  
          # DEBUG:
          # print $fh_out ("% LITERAL: $delimiter$literal$delimiter\n");
  
          $tokens->[@$tokens] = "$delimiter$literal$delimiter";
  
        } elsif ($line =~ /^([0-9]+)([^0-9].*)?$/) {
  
          # An integer literal...
          my $literal = "$1";
          $line = "$2";
  
          # DEBUG:
          # print $fh_out ("% LITERAL: $literal\n");
  
          $tokens->[@$tokens] = $literal;
  
        } elsif ($line =~ /^([a-zA-Z][a-zA-Z0-9_]*)([^a-zA-Z0-9_].*)?$/) {
  
          # An identifier...
          my $identifier = "$1";
          $line = "$2";
  
          # DEBUG:
          # print $fh_out ("% IDENTIFIER: $identifier\n");
  
          $tokens->[@$tokens] = $identifier;
  
        } elsif (($line =~ /^(\+|\*|\?|\(|\)|\:|\;|\|)(.*)$/) and ($in_prologdoc==0)) {#*/ end a prologdoc!!!
  
          # A structural token...
          my $token = "$1";
          $line = "$2";
  
          # DEBUG:
          # print $fh_out ("% TOKEN: $token\n");
  
          $tokens->[@$tokens] = $token;
  
        } elsif ($line =~ /^\{(= )?(.*)$/) {
  
          # Start of Prolog code body...
          $in_code_body = 1;
          $line = "$2";
          # Possibly flag special use of code body...
          if ($1 eq '= ') { $body_is_value = 1; }
          # Skip std lll parsing...
          last;
  
        } elsif ($line =~ /^\<(.*)$/) {
  
          # Start of binding...
          $line = "$1";
  
          # DEBUG:
          # print STDERR ("LT\n");
          $tokens->[@$tokens] = '<';
  
        } elsif ($line =~ /^\>(.*)$/) {
  
          # End of binding...
          $line = "$1";
  
          # DEBUG:
          # print STDERR ("GT\n");
          $tokens->[@$tokens] = '>';
  
        } elsif (($line =~ /^\/\*\*.*$/) and ($in_prologdoc==0)) {
        #} elsif ($line =~ /^\/((\s|\S)*\n)*(\s|\S)*\*\/$/) {
	    #PrologDoc
	    $in_prologdoc=1;
	    # DEBUG:
	    #print STDERR ("% PROLOGDOC-COMMENT START: '$line'\n");

	    $tokens->[@$tokens] = "/**";
	    $line = "$1";

#	    print STDERR ("% PROLOGDOC-COMMENT NEXTLINE: '$line'\n");
        
        } elsif (($line =~ /^\*\/.*$/) and ($in_prologdoc==1)) {
        #} elsif ($line =~ /^\/((\s|\S)*\n)*(\s|\S)*\*\/$/) {
	    #PrologDoc
	    $in_prologdoc=0;
	    # DEBUG:
	    #print STDERR ("% PROLOGDOC-COMMENT END: '$line'\n");

	    $tokens->[@$tokens] = "*/";
	    $line = "$1";

#	    print STDERR ("% PROLOGDOC-COMMENT NEXTLINE: '$line'\n");
        
        } elsif(($line =~ /^.*$/) and ($in_prologdoc==1)) {

	    # DEBUG:
	    #print STDERR ("% PROLOGDOC-COMMENT BULK: '$line'\n");

	    $tokens->[@$tokens] = "$line";
	    $line = "$1";
	    
	} else {
  
          # Scanner error!
          $line =~ /^(.)/;
          scanner_error ("unexpected token in input: '$1' ($line_count)\n");
        }
  
        # Skip whitespace
        $line =~ /^\s*(.*)$/;
        $line = "$1";
      }
    } # End of parsing lll rules.
  } # End of processing input.

  return $line_count, $tokens;
}

sub scan_epilogue {
  my $fh_in = $_[0];
  my $pal_file = $_[1];
  
  my $line_count = 0;
  
  # Current line...
  my $line = '';

  # Current epilogue...
  my $epilogue = '';

  # While end of directives has not been reached...
  while ($line = <$fh_in>) {
    # Bookkeeping...
    $line_count++;
  
    # Just gather up all lines verbatim...
    $epilogue = "$epilogue$line";
  }
  
  $pal_file->{'epilogue'} = $epilogue;
  return $line_count;
}

# ===================================================================
# Parsing...                                                        =
# ===================================================================

sub parse_tokens {
  my $tokens = $_[0];
  my $pal_file = $_[1];
  
  # Current token parsing is at...
  my $position = 0;

  # Bookkeeping...
  my $rule_count = 0;
  
  # While there is more input to process...
  while ($position < @$tokens) {
    # We match a rule starting at the current position...
    my ($len, $rule) = match_rule ($tokens, $position);
  
    if ($len == -1) {
      # No rule found even though we expected one => error in input.
      my $culprit = $tokens->[$position];
      parse_error ("incorrect rule starting with $culprit");
    }
  
    # Add rule...
    $pal_file->add ($rule);
    
    # Bookkeeping...
    $rule_count++;
    
    # Update our position...
    $position += $len;
  }
  $pal_file->{'rule_count_in_input'} = $rule_count;
}

# ===================================================================

sub match_rule {
  # rule : identifier ":" disjunction ";"

  my $tokens = $_[0];
  my $position = $_[1];
  my $len = 0;

  # DEBUG:
  # print STDERR ("RULE ? at $position\n");

  my $rule = PAL_Rule->new ();

  # PROLOGDOC
  if ($tokens->[$position] =~ /^\/\*\*$/) {
      my $prologdoc = "/**\n";
      $len++;
      while(($tokens->[++$position] !~ /^\*\/$/)){
	  $len++;
	  $prologdoc .= "\t" . $tokens->[$position] . "\n";
      }
      $prologdoc .= "*/\n";
    # DEBUG:
    # print STDERR ("PARSED PROLOGDOC:\n'$prologdoc'\n");
      $rule->{'doc'} = $prologdoc;

    $position++; $len++;
  }

  # IDENTIFIER...
  if (not LLL::token_is_identifier($tokens->[$position])) {
    # DEBUG:
    # print STDERR ("RULE: NO IDENTIFIER\n");
    return -1, undef;
  }
  # DEBUG:
  # print STDERR ("RULE: IDENTIFIER '$tokens->[$position]' FOUND\n");

  $rule->{'name'} = $tokens->[$position];
  $position++; $len++;

  # CODE BODY AS VALUE ?
  if ($tokens->[$position] =~ /\{= (.*(\n.*)*) \}/) {
    # DEBUG:
    # print STDERR ("RULE: CODE BODY AS VALUE FOUND\n");
    $rule->{'returns'} = "$1";
    $position++; $len++;
  } else {
    $rule->{'returns'} = '';
  }

  # COLON...
  if (not ($tokens->[$position] =~ /^:$/)) {
    # DEBUG:
    # print STDERR ("RULE: NO COLON\n");
    return -1, undef;
  }
  $position++; $len++;
  # DEBUG:
  # print STDERR ("RULE: COLON FOUND\n");

  # DISJUNCTION...
  # my $disj_len = 0;
  (my $disj_len, $rule->{'body'}) = match_top_disjunction ($tokens, $position);
  if ($disj_len == -1) {
    # DEBUG:
    # print STDERR ("RULE: NO DISJUNCTION\n");
    return -1, undef;
  }
  $position += $disj_len;
  $len += $disj_len;
  # DEBUG:
  # print STDERR ("RULE: DISJUNCTION FOUND\n");

  # SEMICOLON...
  if (not ($tokens->[$position] =~ /^;$/)) {
    # DEBUG:
    # print STDERR ("RULE: NO SEMICOLON\n");
    return -1, undef;
  }
  $position++; $len++;
  # DEBUG:
  # print STDERR ("RULE: SEMICOLON FOUND\n");

  return $len, $rule;
}

# ===================================================================

sub match_top_disjunction {
  # disjunction : { (top_conjunction code_body?) "|" }+

  my $tokens = $_[0];
  my $position = $_[1];
  my $disjunction = PAL_Disjunction->new ();
  my $len = 0;

  # DEBUG:
  # print STDERR ("TOP DISJUNCTION ? at $position\n");

  # CONJUNCTION...
  my ($len_conjunction, $conjunction) = match_top_conjunction($tokens, $position);
  if ($len_conjunction == -1) {
    # DEBUG:
    # print STDERR ("TOP DISJUNCTION: NO CONJUNCTION\n");
    return -1, undef;
  }
  $position += $len_conjunction;
  $len += $len_conjunction;
  # DEBUG:
  # print STDERR ("TOP DISJUNCTION: CONJUNCTION FOUND\n");

  $disjunction->add ($conjunction);

  # CODE BODY ?
  if ($tokens->[$position] =~ /\{ (.*(\n.*)*) \}/) {
    # DEBUG:
    # print STDERR ("TOP DISJUNCTION: CODE BODY FOUND\n");
    $position++; $len++;
    $conjunction->{'code'} = "$1";
  } else {
    # print STDERR ("TOP DISJUNCTION: NO CODE BODY FOUND (that's ok)\n");
    $conjunction->{'code'} = '';
  }

  $more_conjunctions = 1;
  do {
    # OPTION ?
    if ($tokens->[$position] =~ /^\|$/) {
      $more_conjunctions = 1;
      # DEBUG:
      # print STDERR ("TOP DISJUNCTION: OPTION FOUND\n");
      $position++; $len++;
    } else {
      $more_conjunctions = 0;
    }

    if ($more_conjunctions == 1) {
      # CONJUNCTION...
      ($len_conjunction, $conjunction) = match_top_conjunction($tokens, $position);
      if ($len_conjunction == -1) {
        # DEBUG:
        # print STDERR ("TOP DISJUNCTION: NO CONJUNCTION\n");
        return -1, undef;
      }
      $position += $len_conjunction;
      $len += $len_conjunction;
      # DEBUG:
      # print STDERR ("DISJUNCTION: CONJUNCTION FOUND\n");

      $disjunction->add ($conjunction);

      # CODE BODY ?
      if ($tokens->[$position] =~ /\{ (.*(\n.*)*) \}/) {
        # DEBUG:
        # print STDERR ("TOP DISJUNCTION: CODE BODY FOUND\n");
        $position++; $len++;
        $conjunction->{'code'} = "$1";
      } else {
        $conjunction->{'code'} = '';
      }
    }

  } while ($more_conjunctions == 1);

  return $len, $disjunction;
}

# ===================================================================

sub match_top_conjunction {
  # conjunction : (term | "<" identifier ":" term ">")+

  my $tokens = $_[0];
  my $position = $_[1];
  my $conjunction = PAL_Conjunction->new ();
  my $len = 0;
  my $scanning = 1;

  # DEBUG:
  # print STDERR ("TOP CONJUNCTION ? at $position\n");

  do {
    # LT ?
    if ($tokens->[$position] =~ /^\<$/) {
      my $identifier = '';
      my $sub_len = 0;
      # DEBUG:
      # print STDERR ("TOP CONJUNCTION: LT FOUND\n");
      $position++; $sub_len++;

      # IDENTIFIER...
      if (not LLL::token_is_identifier($tokens->[$position])) {
        # DEBUG:
        # print STDERR ("TOP CONJUNCTION: NO IDENTIFIER\n");
        $scanning = 0;
      } else {
        # DEBUG:
        # print STDERR ("TOP CONJUNCTION: IDENTIFIER FOUND\n");
        $identifier = $tokens->[$position];
        $position++; $sub_len++;
      }

      # COLON...
      if ($scanning == 1) {
        if (not ($tokens->[$position] =~ /^:$/)) {
          # DEBUG:
          # print STDERR ("TOP CONJUNCTION: NO COLON\n");
          $scanning = 0;
        } else {
          # DEBUG:
          # print STDERR ("TOP CONJUNCTION: COLON FOUND\n");
          $position++; $sub_len++;
        }
      }

      # TERM...
      my $term;
      if ($scanning == 1) {
        # DEBUG:
        # print STDERR ("TOP CONJUNCTION: TERM ?\n");
        (my $len_term, $term) = LLL::match_term ($tokens, $position);
        if ($len_term == -1) {
          # DEBUG:
          # print STDERR ("TOP CONJUNCTION: NO TERM\n");
          $scanning = 0;
        } else {
          # DEBUG:
          # print STDERR ("TOP CONJUNCTION: TERM FOUND\n");
          
          $position += $len_term;
          $sub_len += $len_term;
        }
      }

      # GT...
      if ($scanning == 1) {
        if (not ($tokens->[$position] =~ /^\>$/)) {
          # DEBUG:
          # print STDERR ("TOP CONJUNCTION: NO GT\n");
          $scanning = 0;
        } else {
          # DEBUG:
          # print STDERR ("TOP CONJUNCTION: GT FOUND\n");
          $position++; $sub_len++;

          $len += $sub_len;
          $term->{'binding'} = $identifier;
          $conjunction->add ($term);
        }
      }

    } else {
      # TERM...
      my ($len_term, $term) = LLL::match_term($tokens, $position);
      if ($len_term == -1) {
        # DEBUG:
        # print STDERR ("TOP CONJUNCTION: NO TERM\n");
        $scanning = 0;
      } else {
        $position += $len_term;
        $len += $len_term;
        # DEBUG:
        # print STDERR ("TOP CONJUNCTION: TERM FOUND\n");

        $conjunction->add ($term);
      }
    }
  } while ($scanning == 1);

  if (@{$conjunction->{'terms'}} == 0) {
    # At least one match is needed...
    # DEBUG:
    # print STDERR ("NO TOP CONJUNCTION\n");
    return -1, undef;
  }

  # DEBUG:
  # print STDERR ("TOP CONJUNCTION ENDS $position\n");
  return $len, $conjunction;
}

# ===================================================================
# Error routines...                                                 =
# ===================================================================

sub scanner_error {
  print STDERR ("Scanner error: $_[0]\n");
  exit (-120);
}

sub parse_error {
  print STDERR ("Parse error: $_[0]\n");
  exit (-140);
}

# ===================================================================
# Classes...                                                        =
# ===================================================================

package PAL_File;

  sub new {
    my($class) = shift;

    return bless {
      'name' => '##id##',
      'starters' => [],
      'rules' => [],
      'epilogue' => ''
    }, $class;
  }

  sub add {
    my $self = shift;
    my $rule = $_[0];

    my $rules = $self->{'rules'};
    $rules->[@{$rules}] = $rule;
  }

  sub as_string {
    my $self = $_[0];
    return "PAL_File (" . $self->{'name'} . ")";
  }

  sub accept {
    my $self = shift;
    my $visitor = $_[0];
    
    if ($visitor->enter_pal_file ($self) == 1) {
      foreach $rule (@{$self->{'rules'}}) {
        $rule->accept ($visitor);
      }
    }
    $visitor->leave_pal_file ($self);
  }

  sub cut_down {
    my $self = shift;

    my $finder = PAL_Active_Rule_Finder->new ();
    $self->accept ($finder);
    $self->{'rules'} = $finder->{'active'};
  }  

  sub complete_using {
    my $self = shift;
    my $lll_grammar = $_[0];
  
    my $my_rules = {};
    foreach $rule (@{$self->{'rules'}}) {
      $my_rules->{$rule->{'name'}} = $rule;
    }

    my $lll_rules = {};
    foreach $rule (@{$lll_grammar->{'rules'}}) {
      $lll_rules->{$rule->{'name'}} = $rule;
    }

    # Basic sanity check...
    my $inconsistencies = [];
    foreach $rule_name (keys %{$my_rules}) {
      if (not $lll_rules->{$rule_name}) {
        $inconsistencies->[@{$inconsistencies}] = $rule_name;
      }
    }
    
    if (@{$inconsistencies} > 0) {
      return $inconsistencies;
    }

    # Completion cycle... 
    foreach $rule_name (keys %{$lll_rules}) {
      if (not $my_rules->{$rule_name}) {
        # DEBUG:
        # print STDERR ("Adding '$rule_name' to PAL.\n");
        $self->{'rules'}->[@{$self->{'rules'}}] = $lll_rules->{$rule_name};
      }
    }
          
    return undef;
  }

# ===================================================================

package PAL_Rule;

  sub new {
    my($class) = shift;

    return bless {
      'name' => 'unnamed rule',
      'returns' => '',
      'body' => undef,
      'doc' => ''
    }, $class;
  }

  sub as_string {
    my $self = $_[0];
    return "PAL_Rule (" . $self->{'name'} . ")";
  }

  sub accept {
    my $self = shift;
    my $visitor = $_[0];

    if ($visitor->enter_pal_rule ($self) == 1) {
	$self->{'body'}->accept ($visitor);
    }
    $visitor->leave_pal_rule ($self);
  }

# ===================================================================

package PAL_Disjunction;

  sub new {
    my($class) = shift;

    return bless {
      'conjunctions' => []
    }, $class;
  }

  sub add {
    my $self = $_[0];
    my $conjunction = $_[1];

    my $conjunctions = $self->{'conjunctions'};
    $conjunctions->[@{$conjunctions}] = $conjunction;
  }

  sub as_string {
    my $self = $_[0];
    return "PAL_Disjunction (" . @{$self->{'conjunctions'}} . ")";
  }

  sub accept {
    my $self = shift;
    my $visitor = $_[0];
    
    if ($visitor->enter_pal_disjunction ($self) == 1) {
      foreach $conjunction (@{$self->{'conjunctions'}}) {
        $conjunction->accept ($visitor);
      }
    }
    $visitor->leave_pal_disjunction ($self);
  }
  
# ===================================================================

package PAL_Conjunction;
use LLL;

  sub new {
    my($class) = shift;

    return bless {
      'terms' => [],
      'code' => ''
    }, $class;
  }

  sub add {
    my $self = $_[0];
    my $term = $_[1];

    my $terms = $self->{'terms'};
    $terms->[@{$terms}] = $term;
  }

  sub as_string {
    my $self = $_[0];
    return "PAL_Conjunction (" . @{$self->{'terms'}} . ")";
  }

  sub accept {
    my $self = shift;
    my $visitor = $_[0];
    
    if ($visitor->enter_pal_conjunction ($self) == 1) {
      foreach $term (@{$self->{'terms'}}) {
        LLL_Term::accept ($term, $visitor);
      }
    }
    $visitor->leave_pal_conjunction ($self);
  }

1;
