#!/usr/bin/perl

# ***** BEGIN LICENSE BLOCK *****
# Version: MPL 1.1/GPL 2.0/LGPL 2.1
#
# The contents of this file are subject to the Mozilla Public License Version
# 1.1 (the "License"); you may not use this file except in compliance with
# the License. You may obtain a copy of the License at
# http://www.mozilla.org/MPL/
#
# Software distributed under the License is distributed on an "AS IS" basis,
# WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
# for the specific language governing rights and limitations under the
# License.
#
# The Original Code is Yerna Lindale.
#
# The Initial Developer of the Original Code is
# the Software Engineering Lab, INTEC, University Ghent.
# Portions created by the Initial Developer are Copyright (C) 2004
# the Initial Developer. All Rights Reserved.
#
# Contributor(s):
#   Kris De Schutter <kris.deschutter@ugent.be>
#
# Alternatively, the contents of this file may be used under the terms of
# either the GNU General Public License Version 2 or later (the "GPL"), or
# the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
# in which case the provisions of the GPL or the LGPL are applicable instead
# of those above. If you wish to allow use of your version of this file only
# under the terms of either the GPL or the LGPL, and not to allow others to
# use your version of this file under the terms of the MPL, indicate your
# decision by deleting the provisions above and replace them with the notice
# and other provisions required by the GPL or the LGPL. If you do not delete
# the provisions above, a recipient may use your version of this file under
# the terms of any one of the MPL, the GPL or the LGPL.
#
# ***** END LICENSE BLOCK *****

package PAL_To_Prolog;

# ===================================================================
# Prolog generator...                                               =
# ===================================================================

  sub new {
    my $class = shift;

    return bless {
      'out' => \*STDOUT,
      'predicate_count' => 0,
      'depth' => 2,
      'term_nr' => 0,
      'term_count' => 0,
      'compound' => '',
      'to_scan' => [],
      'to_match' => [],
      'phrasing' => 0,
      'as_match' => 0
    }, $class;
  }

  sub enter_pal_file {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    my $name = $node->{'name'};
    $self->{'id'} = $name;

    my $starters = $node->{'starters'};

    # For prosperity...
    my $timestamp = localtime ();
    print $out ("% PAL '$name' to Prolog ($timestamp)...\n\n");

    return 1;
  }

  sub leave_pal_file {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    my $id = $self->{'id'};
    my $to_match = $self->{'to_match'};
    my $to_scan = $self->{'to_scan'};

    # If there are extra phrases to be matched...
    if (@{$to_match} > 0) {
      # Output some info...
      print $out ("\n% --------------------------------------------------------------------\n");
      print $out ("% Matching phrases nested in rules...\n");
      print $out ("% --------------------------------------------------------------------\n\n");
    
      $self->{'phrasing'} = 1;
      $self->{'as_match'} = 1;
    
      # For all extra matched phrases...
      for (my $phrase_nr = 0; $phrase_nr < @{$to_match}; $phrase_nr++) {
        # Retrieve the basis for the extra phrase...
        my $basis = $to_match->[$phrase_nr];
    
        # And output the matching code...
        print $out ("match_term($id, phrase$phrase_nr, PAL_1ST, PAL_COMPOSITE) :-\n");

        # generate_phrase_matching (2, $basis);
        # print $out ("  ## TODO ##");
        $basis->accept ($self);

        print $out (",\n");
        print $out ("  !.\n\n");
    
        $self->{'predicate_count'}++;
      } # End of for all extra matched phrases.
    }

    # If there are extra phrases to be scanned...
    if (@{$to_scan} > 0) {
      # Output some info...
      print $out ("\n% --------------------------------------------------------------------\n");
      print $out ("% Scanning phrases nested in rules...\n");
      print $out ("% --------------------------------------------------------------------\n\n");
      
      $self->{'phrasing'} = 1;
      $self->{'as_match'} = 0;
    
      # For all extra matched phrases...
      for (my $phrase_nr = 0; $phrase_nr < @{$to_scan}; $phrase_nr++) {
        # Retrieve the basis for the extra phrase...
        my $basis = $to_scan->[$phrase_nr];
    
        # And output the matching code...
        print $out ("scan_term($id, scanned$phrase_nr, once, PAL_1ST, PAL_NXT) :-\n");

        # generate_phrase_matching (2, $basis);
        # print $out ("  ## TODO ##");
        $basis->accept ($self);

        print $out (",\n");
        print $out ("  !.\n\n");
    
        $self->{'predicate_count'}++;
      } # End of for all extra matched phrases.
    }


    my $epilogue = $node->{'epilogue'};
    if ($epilogue ne '') {
      print $out ("\n% --------------------------------------------------------------------\n");
      print $out ("% Epilogue...\n");
      print $out ("% --------------------------------------------------------------------\n\n");
      print $out ("$epilogue");
    }
  }

  sub enter_pal_rule {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    my $name = $node->{'name'};
    $self->{'rule_name'} = $name;

    my $returns = 'This';
    if ($node->{'returns'} ne '') {
      $returns = $node->{'returns'};
    }
    $self->{'returns'} = $returns;

    my $variation_count = @{$node->{'body'}->{'conjunctions'}};
    print $out ("\n% --------------------------------------------------------------------\n");
    print $out ("% Matching '$name' ($variation_count variation(s))...\n");
    print $out ("% --------------------------------------------------------------------\n\n");

    #ADDED for PrologDoc
    my $doc = $node->{'doc'};

    if ($doc ne "") {
	#my $out = $visitor->{'out'};
	#print STDERR ("ADDED stuff:\n$doc\n");

	#There are PrologDoc-comments...
	if (defined($out)) {
	    print $out ("$doc");
	}
	#print STDERR ("ADDED OK");
    }
    #ADDED for PrologDoc

    return 1;
  }

  sub leave_pal_rule {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};
  }

  sub enter_pal_disjunction {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    return 1;
  }

  sub leave_pal_disjunction {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};
  }

  sub enter_pal_conjunction {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    my $id = $self->{'id'};
    my $rule_name = $self->{'rule_name'};

    my $returns = $self->{'returns'};
    $self->{'predicate_count'}++;

    print $out ("match_term($id, $rule_name, This, $returns) :-\n");
    print $out ("  xml_name(This, '$rule_name'),\n");

    my $depth = $self->{'depth'};
    indented ($out, $depth, "pal_first_element(This, PAL_1ST),\n");

    $self->{'term_nr'} = 0;
    $self->{'compound'} = '';
    $self->{'term_count'} = @{$node->{'terms'}};

    return 1;
  }

  sub leave_pal_conjunction {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    my $depth = $self->{'depth'};

    # Take care of the optional code body...
    my $code_body = $node->{'code'};
    if ($code_body ne '') {
      print $out (",\n");
      indented ($out, $depth + 2, "$code_body");
    }

    print $out (",\n");
    print $out ("  !.\n\n");
  }

  sub enter_term {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    my $depth = $self->{'depth'};
    my $term_nr = $self->{'term_nr'};
    my $term_count = $self->{'term_count'};
    my $repitition = $node->{'repitition'};

    if ($term_nr > 0) {
      print $out (",\n");
    }

    # Construct starting label...
    my $previous = 'PAL_1ST';
    if ($term_nr > 0) {
      my $prev_nr = $term_nr - 1;
      $previous = "PAL_NXT_$prev_nr";
    }

    # Construct finishing label...
    my $next = 'none';
    if ($self->{'phrasing'} == 1) {
      $next = 'PAL_NXT';
    }
    if ($term_nr < $term_count - 1) {
      $next = "PAL_NXT_$term_nr";
    }

    my $id = $self->{'id'};
    my $basis = $node->{'basis'};
    my $type = ref ($basis);

    if (($self->{'as_match'} == 1)
      or ((defined $node->{'binding'}) && ($node->{'binding'} ne ''))) {
      # Term is bound.
      my $binding = $node->{'binding'};
      
      if ($type eq 'LLL_Identifier') {
        # Retrieve identifier...
        my $identifier = $basis->{'value'};
        # Generate step...
        indented ($out, $depth, "match_term($id, $identifier, $repitition, $previous, $next, C$term_nr)");

      } elsif ($type eq 'LLL_Literal') {
        # Retrieve literal...
        my $literal = $basis->{'value'};
        # Make literals single quoted...
        if ($literal =~ /^\"(.*)\"$/) {
          $literal = "'$1'";
        }
        # Generate step...
        indented ($out, $depth, "match_literal($id, $literal, $repitition, $previous, $next, C$term_nr)");

      } elsif ($type eq 'LLL_Disjunction' or $type eq 'LLL_Alternation') {
        # Term is a group.
        # Generate step...
        my $to_match_count = @{$self->{'to_match'}};
        indented ($out, $depth, "match_term($id, phrase$to_match_count, $repitition, $previous, $next, C$term_nr)");
        # Order generation for new phrase...
        $self->{'to_match'}->[@{$self->{'to_match'}}] = $basis;

      } else {
        indented ($out, $depth, "##MATCH $type $repitition##");
      }

      # Have it bind to the prolog result...
      if ($self->{'as_match'} == 0) {
        print $out (",\n");
        indented ($out, $depth + 2, "$binding = C$term_nr");
      } else {
        my $compound = $self->{'compound'};
        # Compose intermediate results...
        if ($compound eq '') {
          $self->{'compound'} = "C$term_nr";
        } else {
          $self->{'compound'} = "$compound, C$term_nr";
        }
      }

    } else {
      # Term is not bound.

      if ($type eq 'LLL_Identifier') {
        # Retrieve identifier...
        my $identifier = $basis->{'value'};
        # Generate step...
        indented ($out, $depth, "scan_term($id, $identifier, $repitition, $previous, $next)");

      } elsif ($type eq 'LLL_Literal') {
        # Retrieve literal...
        my $literal = $basis->{'value'};
        # Make literals single quoted...
        if ($literal =~ /^\"(.*)\"$/) {
          $literal = "'$1'";
        }
        # Generate step...
        indented ($out, $depth, "scan_literal($id, $literal, $repitition, $previous, $next)");

      } elsif ($type eq '%epsilon') {
        indented ($out, $depth, "scan_epsilon($previous, $next),\n");

      } elsif ($type eq 'LLL_Disjunction' or $type eq 'LLL_Alternation') {
        # Term is a group.
        # Generate step...
        my $to_scan_count = @{$self->{'to_scan'}};
        indented ($out, $depth, "scan_term($id, scanned$to_scan_count, $repitition, $previous, $next)");
        # Order generation for new phrase...
        $self->{'to_scan'}->[@{$self->{'to_scan'}}] = $basis;

      } else {
        indented ($out, $depth, "##SCAN $type $repitition##");
      }
    }

    return 0;
  }

  sub leave_term {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    $self->{'term_nr'}++;
  }

  sub enter_disjunction {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    my $depth = $self->{'depth'};
    my $variation_count = @{$node->{'conjunctions'}};

    if ($variation_count == 1) {
      # Singleton group. Phrase == single item.
      my $conjunction = $node->{'conjunctions'}->[0];
      $conjunction->accept ($self);

    } else {
      # Multiple possibilities. Phrase == disjunction.
      # We generate the boilerplating for the options...
      indented ($out, $depth, "(\n");
      for (my $variation_nr = 0; $variation_nr < $variation_count; $variation_nr++) {
        if ($variation_nr > 0) {
          print $out ("\n");
          indented ($out, $depth, ";\n");
        }
        my $conjunction = $node->{'conjunctions'}->[$variation_nr];
        $self->{'depth'} += 2;
        $conjunction->accept ($self);
        $self->{'depth'} -= 2;
      }
      print $out ("\n");
      indented ($out, $depth, ")");
    }

    return 0;
  }

  sub leave_disjunction {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};
  }

  sub enter_conjunction {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    #print $out ("##enter_conjunction##\n");
    $self->{'term_nr'} = 0;
    $self->{'compound'} = '';
    $self->{'term_count'} = @{$node->{'terms'}};
    $self->{'compound'} = '';

    return 1;
  }

  sub leave_conjunction {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    if ($self->{'as_match'} == 1) {
      my $depth = $self->{'depth'};
      my $compound = $self->{'compound'};

      # Generate composed result...
      print $out (",\n");
      if ($compound eq '') {
        # No intermediate results => empty list...
        indented ($out, $depth, "PAL_COMPOSITE = []");
      } elsif ($generated_compound =~ /,/) {
        # Two or more intermediate results => put in list...
        indented ($out, $depth, "PAL_COMPOSITE = [$compound]");
      } else {
        # Only one intermediate result => return as is...
        indented ($out, $depth, "PAL_COMPOSITE = $compound");
      }
    }
    #print $out ("##leave_conjunction##\n");
  }

  sub do_identifier {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};
  }

  sub do_literal {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};
  }

  sub enter_rule {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    my $id = $self->{'id'};

    my $name = $node->{'name'};
    $self->{'rule_name'} = $name;

    my $returns = 'This';
    $self->{'returns'} = $returns;

    # Bookkeeping...
    $self->{'predicate_count'}++;

    print $out ("\n% --------------------------------------------------------------------\n");
    print $out ("% Matching '$name'...\n");
    print $out ("% --------------------------------------------------------------------\n\n");

    # And output the matching code...
    print $out ("match_term($id, $name, This, This) :-\n");
    print $out ("  xml_name(This, '$name'),\n");
    print $out ("  pal_first_element(This, PAL_1ST),\n");

    $self->{'phrasing'} = 1;

    return 1;
  }

  sub leave_rule {
    my $self = shift;
    my $node = $_[0];
    my $out = $self->{'out'};

    $self->{'phrasing'} = 0;

    print $out (",\n");
    print $out ("  PAL_NXT = none,\n");
    print $out ("  !.\n\n");
  }

# ===================================================================
# Methods for walker generation...                                  =
# ===================================================================

sub indented {
  my $out = $_[0];
  my $depth = $_[1];
  my $text = "$_[2]";

  # Every line in $text will be printed with an indentation of
  # $depth number of spaces.

  while ($text =~ /^([^\n]*)\n(.*)$/) {
    for (my $d = $depth; $d > 0; $d--) {
      print $out (' ');
    }
    print $out ("$1\n");
    $text = "$2";
  }

  if ($text ne '') {
    for (my $d = $depth; $d > 0; $d--) {
      print $out (' ');
    }
    print $out ($text);
  }
}

1;
