/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Yerna Lindale.
 *
 * The Initial Developer of the Original Code is
 * the Software Engineering Lab, INTEC, University Ghent.
 * Portions created by the Initial Developer are Copyright (C) 2004
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Kris De Schutter <kris.deschutter@ugent.be>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

/***
	@descr Predicates for working with PAL files.
	@author Kris De Schutter
	@date June 2004
*/

:- load_library('be.ugent.ftw.intec.sel.lillambi.agents.TUPrologDOMXMLLibrary').

we_require('LILLAMBI','lib/xml').

/**
	@form match_term(WID, Identifier, once, First, Next, Compound)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Identifier
        @ground First
        @ground Next
        @unrestricted Compound
	@descr Traverse non-quantized non-terminal in First in lll-production. Afterwards the Next node ((non)-terminal) will be processed and all results are collected in Compound.
*/
match_term(WID, Identifier, once, First, Next, Compound) :-
  match_term(WID, Identifier, First, Compound),
  pal_next_element(First, Next),
  !.

/**
	@form match_term(WID, Identifier, opt, First, Next, Compound)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Identifier
        @ground First
        @ground Next
        @unrestricted Compound
	@descr Traverse ?-quantized non-terminal in First in lll-production. Afterwards the Next node ((non)-terminal) will be processed and all results are collected in Compound.
*/
match_term(WID, Identifier, opt, First, Next, Compound) :-
  match_term(WID, Identifier, once, First, Next, Compound).

match_term(WID, Identifier, opt, First, First, []).

/**
	@form match_term(WID, Identifier, star, First, Next, Compound)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Identifier
        @ground First
        @ground Next
        @unrestricted Compound
	@descr Traverse *-quantized non-terminal in First in lll-production. Afterwards the Next node ((non)-terminal) will be processed and all results are collected in Compound.
*/
match_term(WID, Identifier, star, First, Next, [C|Compound]) :-
  match_term(WID, Identifier, once, First, Intermediate, C),
  match_term(WID, Identifier, star, Intermediate, Next, Compound).

match_term(WID, Identifier, star, First, First, []).

/**
	@form match_term(WID, Identifier, plus, First, Next, Compound)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Identifier
        @ground First
        @ground Next
        @unrestricted Compound
	@descr Traverse +-quantized non-terminal in First in lll-production. Afterwards the Next node ((non)-terminal) will be processed and all results are collected in Compound.
*/
match_term(WID, Identifier, plus, First, Next, [C|Compound]) :-
  match_term(WID, Identifier, once, First, Intermediate, C),
  match_term(WID, Identifier, star, Intermediate, Next, Compound).

/**
	@form match_literal(WID, Literal, once, First, Next, Compound)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Literal
        @ground First
        @ground Next
        @unrestricted Compound
	@descr Traverse non-quantized terminal in First in lll-production. Afterwards the Next node ((non)-terminal) will be processed and all results are collected in Compound.
*/
match_literal(WID, Literal, once, First, Next, Compound) :-
  match_literal(WID, Literal, First, Compound),
  pal_next_element(First, Next),
  !.

/**
	@form match_literal(WID, Literal, opt, First, Next, Compound)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Literal
        @ground First
        @ground Next
        @unrestricted Compound
	@descr Traverse ?-quantized terminal in First in lll-production. Afterwards the Next node ((non)-terminal) will be processed and all results are collected in Compound.
*/
match_literal(WID, Literal, opt, First, Next, Compound) :-
  match_literal(WID, Literal, once, First, Next, Compound).

match_literal(WID, Literal, opt, First, First, []).

/**
	@form match_literal(WID, Literal, star, First, Next, Compound)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Literal
        @ground First
        @ground Next
        @unrestricted Compound
	@descr Traverse *-quantized terminal in First in lll-production. Afterwards the Next node ((non)-terminal) will be processed and all results are collected in Compound.
*/
match_literal(WID, Literal, star, First, Next, [C|Compound]) :-
  match_literal(WID, Literal, once, First, Intermediate, C),
  match_literal(WID, Literal, star, Intermediate, Next, Compound).

match_literal(WID, Literal, star, First, First, []).

/**
	@form match_literal(WID, Literal, plus, First, Next, Compound)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Literal
        @ground First
        @ground Next
        @unrestricted Compound
	@descr Traverse +-quantized terminal in First in lll-production. Afterwards the Next node ((non)-terminal) will be processed and all results are collected in Compound.
*/
match_literal(WID, Literal, plus, First, Next, [C|Compound]) :-
  match_literal(WID, Literal, once, First, Intermediate, C),
  match_literal(WID, Literal, star, Intermediate, Next, Compound).

/**
	@form match_epsilon(First, First, [])
	@constraints
        @ground First
	@descr Traverse empty terminal in First in lll-production. Result is empty list.
*/
match_epsilon(First, First, []).

/**
	@form scan_term(WID, Identifier, once, First, Next)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Identifier
        @ground First
        @ground Next
	@descr Navigate through non-quantized non-terminal in First in lll-production without gathering information. Afterwards the Next node ((non)-terminal) will be processed.
*/
scan_term(WID, Identifier, once, First, Next) :-
  xml_name(First, Identifier),
  pal_next_element(First, Next),
  !.

/**
	@form scan_term(WID, Identifier, opt, First, Next)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Identifier
        @ground First
        @ground Next
	@descr Navigate through ?-quantized non-terminal in First in lll-production without gathering information. Afterwards the Next node ((non)-terminal) will be processed.
*/
scan_term(WID, Identifier, opt, First, Next) :-
  scan_term(WID, Identifier, once, First, Next).

scan_term(WID, Identifier, opt, First, First).

/**
	@form scan_term(WID, Identifier, star, First, Next)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Identifier
        @ground First
        @ground Next
	@descr Navigate through *-quantized non-terminal in First in lll-production without gathering information. Afterwards the Next node ((non)-terminal) will be processed.
*/
scan_term(WID, Identifier, star, First, Next) :-
  scan_term(WID, Identifier, once, First, Intermediate),
  scan_term(WID, Identifier, star, Intermediate, Next).

scan_term(WID, Identifier, star, First, First).

/**
	@form scan_term(WID, Identifier, plus, First, Next)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Identifier
        @ground First
        @ground Next
	@descr Navigate through +-quantized non-terminal in First in lll-production without gathering information. Afterwards the Next node ((non)-terminal) will be processed.
*/
scan_term(WID, Identifier, plus, First, Next) :-
  scan_term(WID, Identifier, once, First, Intermediate),
  scan_term(WID, Identifier, star, Intermediate, Next).

/**
	@form scan_literal(WID, Literal, once, First, Next)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Literal
        @ground First
        @ground Next
	@descr Navigate through non-quantized terminal in First in lll-production without gathering information. Afterwards the Next node ((non)-terminal) will be processed.
*/
scan_literal(WID, Literal, once, First, Next) :-
  scan_literal(WID, Literal, First),
  pal_next_element(First, Next),
  !.

/**
	@form scan_literal(WID, Literal, opt, First, Next)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Literal
        @ground First
        @ground Next
	@descr Navigate through ?-quantized terminal in First in lll-production without gathering information. Afterwards the Next node ((non)-terminal) will be processed.
*/
scan_literal(WID, Literal, opt, First, Next) :-
  scan_literal(WID, Literal, once, First, Next).

scan_literal(WID, Literal, opt, First, First).

/**
	@form scan_literal(WID, Literal, star, First, Next)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Literal
        @ground First
        @ground Next
	@descr Navigate through *-quantized terminal in First in lll-production without gathering information. Afterwards the Next node ((non)-terminal) will be processed.
*/
scan_literal(WID, Literal, star, First, Next) :-
  scan_literal(WID, Literal, once, First, Intermediate),
  scan_literal(WID, Literal, star, Intermediate, Next).

scan_literal(WID, Literal, star, First, First).

/**
	@form scan_literal(WID, Literal, plus, First, Next)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Literal
        @ground First
        @ground Next
	@descr Navigate through +-quantized terminal in First in lll-production without gathering information. Afterwards the Next node ((non)-terminal) will be processed.
*/
scan_literal(WID, Literal, plus, First, Next) :-
  scan_literal(WID, Literal, once, First, Intermediate),
  scan_literal(WID, Literal, star, Intermediate, Next).

/**
	@form scan_epsilon(First, First)
	@constraints
        @ground First
	@descr Navigate through empty terminal in First in lll-production without gathering information.
*/
scan_epsilon(First, First).


/**
	@form pal_first_element(Parent, Element)
	@constraints
        @ground Parent
        @unrestricted Element
	@descr Collect first Element of Parent.
*/
pal_first_element(Parent, Element) :-
  xml_first_child(Parent, Element),
  xml_is_element(Element),  
  !.

pal_first_element(Parent, Element) :-
  xml_first_child(Parent, Child),
  \+ xml_is_element(Child),
  pal_next_element(Child, Element),
  !.

pal_first_element(Parent, none) :-
  !.

/**
	@form pal_next_element(Current, Next)
	@constraints
        @ground Current
        @unrestricted Next
	@descr Look for Next (sibling) element of Current.
*/
pal_next_element(Current, Next) :-
  xml_next(Current, Sibling),
  xml_is_element(Sibling),
  Next = Sibling,
  !.

pal_next_element(Current, Next) :-
  xml_next(Current, Sibling),
  \+ xml_is_element(Sibling),
  pal_next_element(Sibling, Next),
  !.

pal_next_element(Current, Next) :-
  xml_parent(Current, Parent),
  xml_last_child(Parent, Current),
  Next = none,
  !.

/**
	@form pal_previous_element(Current, Previous)
	@constraints
        @ground Current
        @unrestricted Previous
	@descr Look for Previous (sibling) element of Current.
*/
pal_previous_element(Current, Previous) :-
  xml_previous(Current, Sibling),
  xml_is_element(Sibling),
  Previous = Sibling,
  !.

pal_previous_element(Current, Previous) :-
  xml_previous(Current, Sibling),
  \+ xml_is_element(Sibling),
  pal_previous_element(Sibling, Previous),
  !.

pal_previous_element(Current, Previous) :-
  xml_parent(Current, Parent),
  xml_last_child(Parent, Current),
  Previous = none,
  !.

/**
	@form pal_match_all(WID, Name, Root, List)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Name
        @ground Root
        @unrestricted List
	@descr Traverse all nodes rooted at Root in lll-production. All results are collected in List.
*/
pal_match_all(WID, Name, Root, List) :-
  xml_all_named(Name, Root, Nodes),
  pal_match_all_sub(WID, Name, Nodes, List).

/**
	@form pal_match_all_sub(WID, Name, NodeList, List)
	@constraints
	@ground WID ('joinpoints'/'advice')
        @ground Name
        @ground NodeList
        @unrestricted List
	@descr Traverse all nodes from NodeList in lll-production. All results are collected in List.
*/
pal_match_all_sub(_, _, [], []).
pal_match_all_sub(WID, Name, [Head | Tail], [MatchedHead | MatchedTail]) :-
  match_term(WID, Name, Head, MatchedHead),
  pal_match_all_sub(WID, Name, Tail, MatchedTail).

/**
	@form pal_plus_nth_element(Nth, Start, ResultElement)
	@constraints
	@ground Nth
        @ground Start
        @unrestricted ResultElement
	@descr Look for the Nth (sibling) ResultElement after Start (the 0th sibling).
*/
pal_plus_nth_element(0, Start, Start).
pal_plus_nth_element(N, Start, Stop) :-
  pal_next_element(Start, Next),
  M is N - 1,
  pal_plus_nth_element(M, Next, Stop).

/**
	@form pal_minus_nth_element(Nth, Start, ResultElement)
	@constraints
	@ground Nth
        @ground Start
        @unrestricted ResultElement
	@descr Look for the Nth (sibling) ResultElement before Start (the 0th sibling).
*/
pal_minus_nth_element(0, Start, Start).
pal_minus_nth_element(N, Start, Stop) :-
  pal_previous_element(Start, Previous),
  M is N - 1,
  pal_minus_nth_element(M, Previous, Stop).
